<?php
/**
 * @package    JBusinessDirectory
 *
 * @author     CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2021 CMSJunkie. All rights reserved.
 * @license    https://www.gnu.org/licenses/agpl-3.0.en.html
 */
defined('_JEXEC') or die('Restricted access');

require_once BD_CLASSES_PATH . '/attributes/attributeservice.php';

class JBusinessDirectoryControllerMobileConnector extends JControllerLegacy {
	private $appSettings;

	/**
	 * Constructor (registers additional tasks to methods)
	 * @return void
	 * @since 5.0.0
	 */
	public function __construct() {
		$this->appSettings = JBusinessUtil::getApplicationSettings();

		parent::__construct();
	}

	/**
	 * Method to get a model object, loading it if required.
	 *
	 * @param   string $name   The model name. Optional.
	 * @param   string $prefix The class prefix. Optional.
	 * @param   array  $config Configuration array for model. Optional.
	 *
	 * @return  object  The model.
	 *
	 * @since   5.0.0
	 */
	public function getModel($name = 'MobileConnector', $prefix = 'JBusinessDirectoryModel', $config = array('ignore_request' => true)) {
		$model = parent::getModel($name, $prefix, $config);
		return $model;
	}

	/**
	 * Get's search parameters and searches based on the item type. Prints the result
	 * as JSON.
	 *
	 * @since 5.0.0
	 */
	public function getFeaturedItems() {
		$itemType = (int) JFactory::getApplication()->input->get("itemType");

		switch ($itemType) {
			case ITEM_TYPE_EVENT:
				$model = $this->getModel('Events');
				$items = $model->getItems();
				foreach ($items as $item) {
					$item->logoLocation = $item->picture_path;
				}
				break;
			case ITEM_TYPE_OFFER:
                //reset itemType param, same param used in offers structure
                JFactory::getApplication()->input->set("itemType",null);
                $model = $this->getModel('Offers');

                $items = $model->getItems();
				foreach ($items as $item) {
					$item->name         = $item->subject;
					$item->logoLocation = $item->picture_path;
				}
				break;
			default:
				$model = $this->getModel('Search');
				$items = $model->getItems();
		}

		$this->sendResponse($items);
	}

	/**
	 * Get's search parameters and searches based on the item type. Prints the result
	 * as JSON. Filters the result for all 3 types to create a consistent response structure.
	 *
	 * @since 5.0.0
	 */
	public function getSearchResults() {
		$itemType = (int) JFactory::getApplication()->input->get("itemType");
		$itemType = empty($itemType) ? ITEM_TYPE_BUSINESS : $itemType;

		$limitStart = (int) JFactory::getApplication()->input->get("limitstart");
		$limit      = (int) JFactory::getApplication()->input->get("limit");

		if ($limitStart < $limit && $limitStart != 0) {
			$limitStart = $limit;
			JFactory::getApplication()->input->set("limitstart", $limitStart);
		}

		$nameField         = 'name';
		$logoField         = 'logoLocation';
		$mainCategoryField = 'mainSubcategory';
		switch ($itemType) {
			case ITEM_TYPE_EVENT:
				$model             = $this->getModel('Events');
				$items             = $model->getItems();
				$logoField         = 'picture_path';
				$mainCategoryField = 'main_subcategory';
				break;
			case ITEM_TYPE_OFFER:
                //reset itemType param, same param used in offers structure
                JFactory::getApplication()->input->set("itemType",null);
				$model             = $this->getModel('Offers');
				$items             = $model->getItems();
				$nameField         = 'subject';
				$logoField         = 'picture_path';
				$mainCategoryField = 'main_subcategory';
				break;
			default:
				$model = $this->getModel('Search');
				$items = $model->getItems();
		}

		$results = array();
		foreach ($items as $item) {
			$tmp                    = new stdClass();
			$tmp->id                = $item->id;
			$tmp->name              = $item->$nameField;
			$tmp->logo              = $item->$logoField;
			$tmp->short_description = $item->short_description;
			$tmp->longitude         = (float) $item->longitude;
			$tmp->latitude          = (float) $item->latitude;

			$phone = 'N/A';
			if (!empty($item->contact_phone)) {
				$phone = $item->contact_phone;
			} elseif (!empty($item->phone)) {
				$phone = $item->phone;
			} elseif (!empty($item->mobile)) {
				$phone = $item->mobile;
			}

			$tmp->phone         = $phone;
			$tmp->address       = JBusinessUtil::getAddressText($item);
			$tmp->distance      = (float) (isset($item->distance) ? $item->distance : 0);
			$tmp->main_category = (int) $item->$mainCategoryField;

			$tmp->categories = array();
			if (isset($item->categories)) {
				$tmp->categories = $item->categories;
			}

			if ($itemType == ITEM_TYPE_EVENT) {
				$tmp->start_date = $item->start_date;
				$tmp->end_date   = $item->end_date;
			} elseif ($itemType == ITEM_TYPE_OFFER) {
				$tmp->start_date = $item->startDate;
				$tmp->end_date   = $item->endDate;
			}

			if ($itemType == ITEM_TYPE_BUSINESS && !empty($tmp->categories) && !is_array($tmp->categories)) {
				$categories = explode("#|", $tmp->categories);
			} else {
				$categories = !empty($tmp->categories) ? $tmp->categories : array();
			}

			$tmp->categories = array();
			foreach ($categories as $category) {
				$categoryItem = $category;
				if ($itemType == ITEM_TYPE_BUSINESS && !is_array($category)) {
					$categoryItem = explode("|", $category);
				}

				$cat = new stdClass();
				foreach ($categoryItem as $key => $item) {
					switch ($key) {
						case 0:
							$cat->id = (int) $item;
							break;
						case 1:
							$cat->name = $item;
							break;
						case 2:
							$cat->alias = $item;
							break;
					}
				}
				$tmp->categories[] = $cat;
			}

			$tmp->item_type = $itemType;
			$results[]      = $tmp;
		}

		$this->sendResponse($results);
	}

	/**
	 * Provides a suggestion list based on the itemType and searchkeyword parameters.
	 *
	 * @since 5.0.0
	 */
	public function getSuggestions() {
		$itemType = (int) JFactory::getApplication()->input->get("itemType");
		$keyword  = JFactory::getApplication()->input->get("searchkeyword");

		if (empty($itemType)) {
			$itemType = ITEM_TYPE_BUSINESS;
		}

		$model      = $this->getModel();
		$categories = $model->getCategorySuggestions($itemType, $keyword);
		$results    = $model->getSuggestions($itemType, $keyword);

		$items = array();
		foreach ($categories as $category) {
			$category->suggestionType = 2;
			$category->itemType       = $itemType;
			$items[]                  = $category;
		}

		foreach ($results as $result) {
			$result->suggestionType = 1;
			$result->itemType       = $itemType;
			$items[]                = $result;
		}

		$this->sendResponse($items);
	}

	/**
	 * Retrieves company details based on the companyId parameter.
	 *
	 * @since 5.0.0
	 */
	public function getCompany() {
		$appSettings = JBusinessUtil::getApplicationSettings();
		$token       = JFactory::getApplication()->input->get("token");

		$user     = null;
		$apiModel = $this->getModel();
		if ($this->validateToken() && !empty($token)) {
			$user = $apiModel->getUserByToken($token);
		}

		$attributeConfig = JBusinessUtil::getAttributeConfiguration(DEFAULT_ATTRIBUTE_TYPE_LISTING);
		$showAddress     = $attributeConfig["street_number"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["address"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["area"] != ATTRIBUTE_NOT_SHOW
			|| $attributeConfig["country"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["city"] != ATTRIBUTE_NOT_SHOW
			|| $attributeConfig["province"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["region"] != ATTRIBUTE_NOT_SHOW
			|| $attributeConfig["postal_code"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["map"] != ATTRIBUTE_NOT_SHOW;

		$model            = $this->getModel('Companies');
		$company          = $model->getCompany();
		$company->package = $model->getPackage($company->id);

		$model->increaseViewCount($company->id);

		$images = array();
		if ($attributeConfig["pictures"] != ATTRIBUTE_NOT_SHOW) {
			$images = $model->getCompanyImages();
		}

		$company->images = array();
		foreach ($images as $image) {
			$company->images[] = $image->picture_path;
		}

		if (count($company->images) == 0) {
			$company->images[]     = '/no_image.jpg';
			$company->logoLocation = '/no_image.jpg';
		}

		$company->id            = (int) $company->id;
		$company->longitude     = (float) $company->longitude;
		$company->latitude      = (float) $company->latitude;
		$company->review_score  = (float) $company->review_score;
		$company->averageRating = (float) $company->averageRating;
		$company->viewCount     = (int) $company->viewCount;
		$company->userId        = (int) $company->userId;
		$company->reviews       = $model->getReviews();
		$company->reviews_count = count($company->reviews);

		$address          = JBusinessUtil::getAddressText($company);
		$company->address = !empty($address) ? $address : null;
		if (!$showAddress) {
			$company->address = null;
		}

		$phone = $company->phone;
		if (empty($phone)) {
			$phone = $company->mobile;
			if (empty($phone)) {
				$contacts = $model->getCompanyContacts();

				if (!empty($contacts) && !empty($contacts[0]->contact_phone)) {
					$phone = $contacts[0]->contact_phone;
				}
			}
		}

		$company->phone    = $phone;
		$company->shareUrl = JBusinessUtil::getCompanyLink($company);

		$hours             = array();
		$businessHours     = $company->business_hours;
		$company->isOpened = false;
		$dayCount          = 1;
		$hasHours          = false;
		foreach ($businessHours as $day) {
			$tmp                      = new stdClass();
			$tmp->workHours           = $day->workHours;
			$tmp->workHours['status'] = (int) $tmp->workHours['status'];
			$tmp->workHours['id']     = (int) $tmp->workHours['id'];

			$curDay = date('N');
			if ($dayCount == $curDay) {
				if ($tmp->workHours['status'] == 1) {
					if (empty($tmp->workHours['start_time']) && empty($tmp->workHours['end_time'])) {
						$company->isOpened = true;
					} else {
						$start = strtotime(date('Y-m-d') . ' ' . $tmp->workHours['start_time']);
						$end   = strtotime(date('Y-m-d') . ' ' . $tmp->workHours['end_time']);
						$now   = strtotime(date('Y-m-d H:i:s'));

						if ($start <= $now && $now <= $end) {
							$company->isOpened = true;
						}
					}
				}
			}

			if (!empty($tmp->workHours['start_time']) || !empty($tmp->workHours['end_time'])) {
				$hasHours = true;
			}

			$tmp->workHours['start_time'] = !empty($tmp->workHours['start_time']) && $tmp->workHours['status'] == 1 ? JBusinessUtil::convertTimeToFormat($tmp->workHours['start_time']) : null;
			$tmp->workHours['end_time']   = !empty($tmp->workHours['end_time']) && $tmp->workHours['status'] == 1 ? JBusinessUtil::convertTimeToFormat($tmp->workHours['end_time']) : null;
			$tmp->name                    = $day->name;

			$hours[] = $tmp;
		}
		$company->business_hours = ($hasHours) ? $hours : null;

		foreach ($company->reviews as $review) {
			$review->rating = (float) $review->rating;
			$review->userId = (int) $review->userId;
		}

		$company->isBookmarked = false;
		if (!empty($user)) {
			$company->isBookmarked = $apiModel->isBookmarked($user->id, $company->id);
		}

		$company->isOpened  = true;
		$company->hasDevice = false;
		$userId             = $company->userId;
		if (!empty($userId)) {
			$apiModel = $this->getModel();
			$devices  = $apiModel->getDevicesByUser($userId);
			if (count($devices) > 0) {
				$company->hasDevice = true;
			}
		}

		$company->contact_link = JBusinessUtil::getCompanyLink($company);

		$this->package = $model->getPackage();

		$company->attributes = $model->getCompanyAttributes();
		if ((isset($this->package->features) && in_array(SERVICES_LIST, $this->package->features) || !$appSettings->enable_packages)) {
			$company->services = $model->getServicesList();
		}
		$company->description = $this->prepareCompanyDetails($company);

		$this->sendResponse($company);
	}

	/**
	 * Retrieves events for a certain company by companyId
	 *
	 * @since 5.1.3
	 */
	public function getCompanyEvents() {
		$model  = $this->getModel('Companies');
		$events = $model->getCompanyEvents();

		foreach ($events as $item) {
			$item->logoLocation = $item->picture_path;
		}


		$this->sendResponse($events);
	}

	/**
	 * Retrieves offers for a certain company by companyId
	 *
	 * @since 5.1.3
	 */
	public function getCompanyOffers() {
		$model  = $this->getModel('Companies');
		$offers = $model->getCompanyOffers();

		foreach ($offers as $item) {
			$item->name         = $item->subject;
			$item->logoLocation = $item->picture_path;
		}

		$this->sendResponse($offers);
	}

	/**
	 * Retrieves event details based on the eventId parameter.
	 *
	 * @since 5.0.0
	 */
	public function getEvent() {
		$model = $this->getModel('Event');
		$event = $model->getEvent();

		$model->increaseViewCount($event->id);

		$attributeConfig = JBusinessUtil::getAttributeConfiguration(DEFAULT_ATTRIBUTE_TYPE_EVENT);
		$showAddress     = $attributeConfig["street_number"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["address"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["area"] != ATTRIBUTE_NOT_SHOW
			|| $attributeConfig["country"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["city"] != ATTRIBUTE_NOT_SHOW
			|| $attributeConfig["province"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["region"] != ATTRIBUTE_NOT_SHOW
			|| $attributeConfig["postal_code"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["map"] != ATTRIBUTE_NOT_SHOW;

		$event->images = array();
		foreach ($event->pictures as $picture) {
			$event->images[] = $picture->picture_path;
		}

		if (count($event->images) == 0) {
			$event->images[] = '/no_image.jpg';
		}

		$event->id           = (int) $event->id;
		$event->logoLocation = isset($event->images[0]) ? $event->images[0] : '';
		$event->phone        = !empty($event->contact_phone) ? $event->contact_phone : $event->company->phone;
		$event->longitude    = (float) $event->longitude;
		$event->latitude     = (float) $event->latitude;
		$event->userId       = (int) $event->user_id;
		$event->viewCount    = (int) $event->view_count;

		$startDate = null;
		if (!JBusinessUtil::emptyDate($event->start_date)) {
			$startDate = JBusinessUtil::getDateGeneralFormat(date('Y-m-d', strtotime($event->start_date)));
		}

		$endDate = null;
		if (!JBusinessUtil::emptyDate($event->end_date) && $event->show_end_date) {
			$endDate = JBusinessUtil::getDateGeneralFormat(date('Y-m-d', strtotime($event->end_date)));
		}

		$event->startDate = $startDate;
		$event->endDate   = $endDate;
		$event->website   = $event->company->website;

		$event->startTime = (!empty($event->start_time) && $event->show_start_time) ? $event->start_time : null;
		$event->endTime   = (!empty($event->end_time) && $event->show_end_time) ? $event->end_time : null;

		$address        = JBusinessUtil::getAddressText($event);
		$event->address = !empty($address) ? $address : null;
		if (!$showAddress) {
			$event->address = null;
		}

		$event->alias    = isset($event->alias) ? $event->alias : '';
		$event->shareUrl = JBusinessUtil::getEventLink($event->id, $event->alias);

		$event->isOpened  = true;
		$event->hasDevice = false;
		$userId           = !empty($event->user_id) ? $event->user_id : $event->company->userId;
		if (!empty($userId)) {
			$apiModel = $this->getModel();
			$devices  = $apiModel->getDevicesByUser($userId);
			if (count($devices) > 0) {
				$event->hasDevice = true;
			}
		}

		$event->company_name = $event->company->name;
		$event->company_id   = (int) $event->company_id;
		$event->contact_link = JBusinessUtil::getEventLink($event->id, $event->alias);

		$event->description = $this->prepareEventDetails($event);

		$this->sendResponse($event);
	}

	/**
	 * Retrieves offer details based on the offerId parameter.
	 *
	 * @since 5.0.0
	 */
	public function getOffer() {
		$token = JFactory::getApplication()->input->get("token");

		$user     = null;
		$apiModel = $this->getModel();
		if ($this->validateToken() && !empty($token)) {
			$user = $apiModel->getUserByToken($token);
		}

		$model = $this->getModel('Offer');
		$offer = $model->getOffer();

		$model->increaseViewCount($offer->id);

		$attributeConfig = JBusinessUtil::getAttributeConfiguration(DEFAULT_ATTRIBUTE_TYPE_OFFER);
		$showAddress     = $attributeConfig["street_number"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["address"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["area"] != ATTRIBUTE_NOT_SHOW
			|| $attributeConfig["country"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["city"] != ATTRIBUTE_NOT_SHOW
			|| $attributeConfig["province"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["region"] != ATTRIBUTE_NOT_SHOW
			|| $attributeConfig["postal_code"] != ATTRIBUTE_NOT_SHOW || $attributeConfig["map"] != ATTRIBUTE_NOT_SHOW;


		$offer->images = array();
		foreach ($offer->pictures as $picture) {
			$offer->images[] = $picture->picture_path;
		}

		if (count($offer->images) == 0) {
			$offer->images[] = '/no_image.jpg';
		}

		$offer->id           = (int) $offer->id;
		$offer->name         = $offer->subject;
		$offer->logoLocation = isset($offer->images[0]) ? $offer->images[0] : '';
		$offer->price        = (float) $offer->price;
		if (empty($offer->price)) {
			$offer->priceText = JText::_('LNG_FREE');
		} else {
			$offer->priceText    = JBusinessUtil::getPriceFormat($offer->price, $offer->currencyId);
		}
		$offer->longitude    = (float) $offer->longitude;
		$offer->latitude     = (float) $offer->latitude;
		$offer->userId       = (int) $offer->user_id;
		$offer->viewCount    = (int) $offer->viewCount;
		$offer->website      = $offer->company->website;
		$offer->phone        = $offer->company->phone;
		$offer->startDate    = isset($offer->startDate) ? $offer->startDate : '';
		$offer->endDate      = isset($offer->endDate) ? $offer->endDate : '';

		$startDate = null;
		if (!JBusinessUtil::emptyDate($offer->startDate)) {
			$startDate = JBusinessUtil::getDateGeneralFormat(date('Y-m-d', strtotime($offer->startDate)));
		}

		$endDate = null;
		if (!JBusinessUtil::emptyDate($offer->endDate)) {
			$endDate = JBusinessUtil::getDateGeneralFormat(date('Y-m-d', strtotime($offer->endDate)));
		}

		$offer->startDate = $startDate;
		$offer->endDate   = $endDate;

		$address        = JBusinessUtil::getAddressText($offer);
		$offer->address = !empty($address) ? $address : null;
		if (!$showAddress) {
			$offer->address = null;
		}

		$offer->reviews = $model->getReviews();

		$totScore            = 0.0;
		$offer->review_score = 0.0;
		if (count($offer->reviews) > 0) {
			foreach ($offer->reviews as $review) {
				$totScore += (float) $review->rating;
			}

			$offer->review_score = (float) ($totScore / (float) count($offer->reviews));
		}

		$offer->alias    = isset($offer->alias) ? $offer->alias : '';
		$offer->shareUrl = JBusinessUtil::getOfferLink($offer->id, $offer->alias);

		$offer->isBookmarked = false;
		if (!empty($user)) {
			$offer->isBookmarked = $apiModel->isBookmarked($user->id, $offer->id, BOOKMARK_TYPE_OFFER);
		}

		$offer->isOpened = true;
		if (!empty($offer->startDate) && !empty($offer->endDate)) {
			if (!JBusinessUtil::checkDateInterval($offer->startDate, $offer->endDate)) {
				$offer->isOpened = false;
			}
		} elseif (!empty($offer->endDate)) {
			if (strtotime(date('Y-m-d')) > strtotime($offer->endDate)) {
				$offer->isOpened = false;
			}
		} elseif (!empty($offer->startDate)) {
			if (strtotime(date('Y-m-d')) < strtotime($offer->startDate)) {
				$offer->isOpened = false;
			}
		}

		$offer->hasDevice = false;
		$userId           = !empty($offer->user_id) ? $offer->user_id : $offer->company->userId;
		if (!empty($userId)) {
			$apiModel = $this->getModel();
			$devices  = $apiModel->getDevicesByUser($userId);
			if (count($devices) > 0) {
				$offer->hasDevice = true;
			}
		}

		$offer->company_name = $offer->company->name;
		$offer->company_id   = (int) $offer->companyId;
		$offer->contact_link = JBusinessUtil::getOfferLink($offer->id, $offer->alias);

		$offer->description = $this->prepareOfferDetails($offer);

		$this->sendResponse($offer);
	}

	/**
	 * Retrieves the company/event types and prints them as JSON.
	 *
	 * @since 5.0.0
	 */
	public function getTypes() {
		$itemType = (int) JFactory::getApplication()->input->get("itemType");

		$model = $this->getModel();
		switch ($itemType) {
			case ITEM_TYPE_OFFER:
				$items = $model->getOfferTypes();
				break;
			case ITEM_TYPE_EVENT:
				$items = $model->getEventTypes();
				break;
			default:
				$items = $model->getCompanyTypes();
		}

		$tmp       = new stdClass();
		$tmp->id   = '0';
		$tmp->name = JText::_('LNG_ALL_TYPES');
		array_unshift($items, $tmp);

		$this->sendResponse($items);
	}

	/**
	 * Retrieves the company/event/offer categories and prints them as JSON.
	 *
	 * @since 5.0.0
	 */
	public function getCategories() {
		$itemType = (int) JFactory::getApplication()->input->get("itemType");

		$model = $this->getModel();
		$items = $model->getCategories($itemType);

		$this->sendResponse($items);
	}

	/**
	 * Retrieves the sort by configuration based on item type.
	 *
	 * @since 5.0.0
	 */
	public function getSortByConfiguration() {
		$itemType = (int) JFactory::getApplication()->input->get("itemType");

		switch ($itemType) {
			case ITEM_TYPE_OFFER:
				$model = $this->getModel('Offers');
				$items = $model->getSortByConfiguration();
				break;
			case ITEM_TYPE_EVENT:
				$model = $this->getModel('Events');
				$items = $model->getSortByConfiguration();
				break;
			default:
				$model = $this->getModel('Search');
				$items = $model->getSortByConfiguration();
		}

		$this->sendResponse($items);
	}

	/**
	 * Retrieves reviews belonging to a certain company or offer (defined by itemType).
	 *
	 * @since 5.0.0
	 */
	public function getReviews() {
		$itemId   = (int) JFactory::getApplication()->input->get("itemId");
		$itemType = (int) JFactory::getApplication()->input->get("itemType");

		switch ($itemType) {
			case ITEM_TYPE_OFFER:
				JFactory::getApplication()->input->set("offerId", $itemId);
				$model = $this->getModel('Offer');
				$items = $model->getReviews();
				break;
			default:
				JFactory::getApplication()->input->set("companyId", $itemId);
				$model = $this->getModel('Companies');
				$items = $model->getReviews();
				foreach ($items as $item) {
					if (isset($item->criteriaIds)) {
						$item->criterias = $model->getReviewCriterias();
					}
				}
		}

		$this->sendResponse($items);
	}

	/**
	 * Retrieves parameters and creates a new review
	 *
	 * @since 5.0.0
	 */
	public function addReview() {
		$itemType = (int) JFactory::getApplication()->input->get("itemType");
		$data     = JFactory::getApplication()->input->getArray();

		switch ($itemType) {
			case ITEM_TYPE_OFFER:
				$model               = $this->getModel('Offer');
				$data["review_type"] = REVIEW_TYPE_OFFER;
				break;
			default:
				$model               = $this->getModel('Companies');
				$data["review_type"] = REVIEW_TYPE_BUSINESS;
		}

		$userId = $data["userId"];
		$itemId = $data["itemId"];

		if ($data['itemUserId'] == $data["userId"]) {
			$result = array(0 => JText::_('LNG_NO_REVIEW_FROM_ITEM_OWNER'));
			$this->sendResponse($result, RESPONSE_STATUS_ERROR);
			return;
		}

		if ($model->checkUserReviews($userId, $itemId)) {
			$result = array(0 => JText::_('LNG_NO_MORE_THAN_ONE_REVIEW'));
			$this->sendResponse($result, RESPONSE_STATUS_ERROR);
			return;
		}

		if (!isset($data["rating"])) {
			$data["rating"] = $data['review'];
		}
		
		switch ($itemType) {
			case ITEM_TYPE_OFFER:
				$result = $model->saveReview($data);
				break;
			default:
				$companyId = JFactory::getApplication()->input->get('itemId');
				$company = $model->getPlainCompany($itemId);
				$result = $model->saveReview($data,$company);
		}
		
		

		$this->sendResponse($result, RESPONSE_STATUS_SUCCESS);
	}

	/**
	 * Adds a new bookmark for a certain company/offer based on
	 * item_id and item_type.
	 *
	 * @since 5.0.0
	 */
	public function addBookmark() {
		$data = JFactory::getApplication()->input->getArray(
			array(
				'item_id'   => 'int',
				'user_id'   => 'int',
				'note'      => 'raw',
				'item_type' => 'int'
			)
		);

		$itemType = $data["item_type"];
		switch ($itemType) {
			case ITEM_TYPE_OFFER:
				$model    = $this->getModel('Offer');
				$response = $model->addBookmark($data);
				break;
			default:
				$model    = $this->getModel('Companies');
				$response = $model->addBookmark($data);
		}

		NotificationService::sendBookmarkNotification($data["item_id"], $data["item_type"]);
		$this->sendResponse($response);
	}

	/**
	 * Updates a bookmark based on it's id.
	 *
	 * @since 5.0.0
	 */
	public function updateBookmark() {
		$data = JFactory::getApplication()->input->getArray(
			array(
				'id'        => 'int',
				'note'      => 'raw',
				'item_type' => 'int'
			)
		);

		$itemType = $data["item_type"];
		switch ($itemType) {
			case ITEM_TYPE_OFFER:
				$model    = $this->getModel('Offer');
				$response = $model->updateBookmark($data);
				break;
			default:
				$model    = $this->getModel('Companies');
				$response = $model->updateBookmark($data);
		}

		$this->sendResponse($response);
	}

	/**
	 * Removes a bookmark based on the submitted id.
	 *
	 * @since 5.0.0
	 */
	public function removeBookmark() {
		$data = JFactory::getApplication()->input->getArray(
			array(
				'item_id'   => 'int',
				'item_type' => 'int',
				'user_id'   => 'int'
			)
		);

		$model  = $this->getModel();
		$result = $model->removeBookmark($data);

		$this->sendResponse($result);
	}

	/**
	 * Retrieves the user bookmarks and prints them as JSON.
	 *
	 * @since 5.0.0
	 */
	public function getUserBookmarks() {
		$userId = (int) JFactory::getApplication()->input->get("userId");
		if (!$this->validateToken()) {
			$this->sendResponse(JText::_('LNG_INVALID_TOKEN'), RESPONSE_STATUS_INVALID_TOKEN);
		}

		$model = $this->getModel();
		$items = $model->getUserBookmarks($userId);

		$this->sendResponse($items);
	}

	/**
	 * Retrieves the user bookmarks and prints them as JSON.
	 *
	 * @since 5.0.0
	 */
	public function getUserReviews() {
		$userId = (int) JFactory::getApplication()->input->get("userId");
		if (!$this->validateToken()) {
			$this->sendResponse(JText::_('LNG_INVALID_TOKEN'), RESPONSE_STATUS_INVALID_TOKEN);
		}

		$appSettings = JBusinessUtil::getApplicationSettings();
		$items       = array();
		if ($appSettings->enable_reviews) {
			$model = $this->getModel();
			$items = $model->getUserReviews($userId);
		}

		$this->sendResponse($items);
	}

	/**
	 * Enables/disables push notification setting of a certain device (token) based on the
	 * value of the enable parameter.
	 *
	 * @since 5.0.0
	 */
	public function setPushNotifications() {
		$enable = (int) JFactory::getApplication()->input->get("enable");
		$token  = JFactory::getApplication()->input->get("token");

		if (!$this->validateToken()) {
			$this->sendResponse(JText::_('LNG_INVALID_TOKEN'), RESPONSE_STATUS_INVALID_TOKEN);
		}

		$model  = $this->getModel();
		$result = $model->setPushNotifications($enable, $token);

		$this->sendResponse($result);
	}

	/**
	 * Sets the firebase registration token for a specific device based on its session token.
	 *
	 * @since 5.0.0
	 */
	public function setFirebaseToken() {
		$token    = JFactory::getApplication()->input->get("token", '', 'RAW');
		$firebase = JFactory::getApplication()->input->get("firebase_token", '', 'RAW');

		if (!$this->validateToken()) {
			$this->sendResponse(JText::_('LNG_INVALID_TOKEN'), RESPONSE_STATUS_INVALID_TOKEN);
		}

		$model  = $this->getModel();
		$result = $model->setFirebaseToken($token, $firebase);

		$this->sendResponse($result);
	}

	/**
	 * Retrieves the username and password from the request, log's the user in and
	 * creates a record for the user and device and returns the session token.
	 *
	 * @since 5.0.0
	 */
	public function logIn() {
		$username = JFactory::getApplication()->input->get("username", '', 'USERNAME');
		$password = JFactory::getApplication()->input->get("password", '', 'RAW');
		$deviceId = JFactory::getApplication()->input->get("deviceId", '', 'RAW');

		$credentials             = array();
		$credentials['username'] = $username;
		$credentials['password'] = $password;

		$app    = JFactory::getApplication();
		$result = $app->login($credentials);

		// login failed
		if (!$result) {
			$result = array(0 => JText::_('LNG_LOGIN_FAILED'));
			$this->sendResponse($result, RESPONSE_STATUS_ERROR);
		} else {
			$userId = JBusinessUtil::getUser()->id;

			$model = $this->getModel();
			$token = $model->saveMobileUser($userId, $deviceId);

			$user                 = JBusinessUtil::getUser();
			$user->token          = $token;
			$user->password_clear = $password;

			// if false, than error has occurred
			if (!$token) {
				$this->sendResponse($token, RESPONSE_STATUS_ERROR);
			}

			// send token
			$this->sendResponse($user);
		}
	}

	/**
	 * Registers new user based on the request parameters. Returns user ID on success.
	 *
	 * @since 5.0.0
	 */
	public function register() {
		$name     = JFactory::getApplication()->input->get("name", '', 'USERNAME');
		$username = JFactory::getApplication()->input->get("username", '', 'USERNAME');
		$password = JFactory::getApplication()->input->get("password", '', 'RAW');
		$email    = JFactory::getApplication()->input->get("email", '', 'RAW');

		$data = array(
			"name"      => $name,
			"username"  => $username,
			"password"  => $password,
			"password2" => $password,
			"email"     => $email,
			"block"     => 0,
			"groups"    => array($this->appSettings->mobile_usergroup)
		);

		// register user
		$user = new JUser;
		if (!$user->bind($data)) {
			$this->sendResponse($user->getError(), RESPONSE_STATUS_ERROR);
		}
		if (!$user->save()) {
			$this->sendResponse($user->getError(), RESPONSE_STATUS_ERROR);
		}

		try {
			JUserHelper::addUserToGroup($user->id, $this->appSettings->mobile_usergroup);
		} catch (Exception $e) {
			$this->sendResponse($e->getMessage(), RESPONSE_STATUS_ERROR);
		}

		$this->logIn();
	}

	/**
	 * Logs out a device by deleting its record from the database based on the token
	 * received.
	 *
	 * @since 5.0.0
	 */
	public function logOut() {
		$token = JFactory::getApplication()->input->get("token");
		if (!$this->validateToken()) {
			$this->sendResponse(JText::_('LNG_INVALID_TOKEN'), RESPONSE_STATUS_INVALID_TOKEN);
		}

		$model  = $this->getModel();
		$result = $model->logOut($token);

		$this->sendResponse($result);
	}

	/**
	 * Updates the user (identified by the token) with new values
	 *
	 * @since 5.0.0
	 */
	public function updateProfile() {
		$name     = JFactory::getApplication()->input->get("name", '', 'USERNAME');
		$username = JFactory::getApplication()->input->get("username", '', 'USERNAME');
		$password = JFactory::getApplication()->input->get("password", '', 'RAW');
		$email    = JFactory::getApplication()->input->get("email", '', 'RAW');
		$token    = JFactory::getApplication()->input->get("token");

		if (!$this->validateToken()) {
			$this->sendResponse(JText::_('LNG_INVALID_TOKEN'), RESPONSE_STATUS_INVALID_TOKEN);
		}

		$data = array(
			"name"      => $name,
			"username"  => $username,
			"password"  => $password,
			"password2" => $password,
			"email"     => $email,
			"block"     => 0,
		);

		$model                = $this->getModel();
		$user                 = $model->getUserByToken($token);
		$user->token          = $token;
		$user->password_clear = $password;

		$groups         = JUserHelper::getUserGroups($user->id);
		$data['groups'] = $groups;

		if (!$user->bind($data)) {
			$this->sendResponse($user->getError(), RESPONSE_STATUS_ERROR);
		}
		if (!$user->save()) {
			$this->sendResponse($user->getError(), RESPONSE_STATUS_ERROR);
		}

		$this->sendResponse($user, RESPONSE_STATUS_SUCCESS);
	}

	/**
	 * Logs the user and loads the control panel screen if login is successful.
	 *
	 * @since 5.0.0
	 */
	public function showControlPanel() {
		$token = JFactory::getApplication()->input->get("token");

		if (!$this->validateToken()) {
			$this->sendResponse(JText::_('LNG_INVALID_TOKEN'), RESPONSE_STATUS_INVALID_TOKEN);
		}

		$model   = $this->getModel();
		$userObj = $model->getUserByToken($token);

		$user             = array();
		$user['id']       = $userObj->id;
		$user['name']     = $userObj->name;
		$user['username'] = $userObj->username;
		$user['password'] = $userObj->password;

		JPluginHelper::importPlugin('user');
		$dispatcher = JDispatcher::getInstance();

		// Initiate log in
		$options = array('action' => 'core.login.site', 'remember' => false);
		if ($dispatcher->trigger('onUserLogin', array($user, $options))) {
			$this->setRedirect(JRoute::_('index.php?option=com_jbusinessdirectory&view=useroptions&tmpl=component', false));
		} else {
			$this->sendResponse(false, RESPONSE_STATUS_ERROR);
		}
	}


    /**
     * Logs the user and loads the control panel screen if login is successful.
     *
     * @since 5.0.0
     */
    public function showUserPackages() {
        $token = JFactory::getApplication()->input->get("token");

        if (!$this->validateToken()) {
            $this->sendResponse(JText::_('LNG_INVALID_TOKEN'), RESPONSE_STATUS_INVALID_TOKEN);
        }

        $model   = $this->getModel();
        $userObj = $model->getUserByToken($token);

        $user             = array();
        $user['id']       = $userObj->id;
        $user['name']     = $userObj->name;
        $user['username'] = $userObj->username;
        $user['password'] = $userObj->password;

        JPluginHelper::importPlugin('user');
        $dispatcher = JDispatcher::getInstance();

        // Initiate log in
        $options = array('action' => 'core.login.site', 'remember' => false);
        if ($dispatcher->trigger('onUserLogin', array($user, $options))) {
            $this->setRedirect(JRoute::_('index.php?option=com_jbusinessdirectory&view=packages&packageType=2&tmpl=component', false));
        } else {
            $this->sendResponse(false, RESPONSE_STATUS_ERROR);
        }
    }

	/**
	 * Returns the terms and conditions text defined in general settings
	 *
	 * @since 5.0.0
	 */
	public function getTermsAndConditions() {
		$appSettings = JBusinessUtil::getApplicationSettings();
		$terms       = $appSettings->terms_conditions;

		echo $terms;
	}

	/**
	 * Retrieves ID of the message sender, ID of the receiver, subject and body of message, and sends
	 * a notification to the receiver's device about the message.
	 *
	 * @since 5.0.0
	 */
	public function sendMessageNotification() {
		$senderId   = JFactory::getApplication()->input->getInt("senderId");
		$receiverId = JFactory::getApplication()->input->getInt("receiverId");
		$subject    = JFactory::getApplication()->input->get("subject", '', 'RAW');
		$body       = JFactory::getApplication()->input->get("body", '', 'RAW');

		if (!$this->validateToken()) {
			$this->sendResponse(JText::_('LNG_INVALID_TOKEN'), RESPONSE_STATUS_INVALID_TOKEN);
		}

		$response = NotificationService::sendMessageNotification($senderId, $receiverId, $subject, $body);

		$status = RESPONSE_STATUS_SUCCESS;
		if (!$response) {
			$status = RESPONSE_STATUS_ERROR;
		}

		$this->sendResponse($response, $status);
	}

	/**
	 * Prepares the company info to be shown on company details screen
	 *
	 * @param $company object
	 *
	 * @return string
	 *
	 * @since 5.1.0
	 */
	public function prepareCompanyDetails($company) {
		$appSettings = JBusinessUtil::getApplicationSettings();

		$details = '';
		if (!empty($company->mainCategory)) {
			$details .= '<b>' . JText::_('LNG_CATEGORY') . '</b>: ';
			$details .= $company->mainCategory . '<br/><br/>';
		}

		$packageFeatured = isset($this->package->features) ? $this->package->features : null;

		$details .= AttributeService::renderAttributesFront($company->attributes, $appSettings->enable_packages, $packageFeatured);
		$details .= '<br/>';

		if (isset($company->services) && count($company->services) > 0) {
			$header  = $company->services[0]->service_section;
			$details .= '<h4>' . $company->services[0]->service_section . '</h4>';
			foreach ($company->services as $key => $service) {
				if ($header != $service->service_section) {
					$details .= '<h4>' . $service->service_section . '</h4>';
				}
				$details .= '<b>' . $service->service_name . '</b>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;' . JBusinessUtil::getPriceFormat($service->service_price, $appSettings->currency_id) . '<br/>';
				$details .= $service->service_description . '<br/><br/>';
				$header  = $service->service_section;
			}
			$details .= '<br/><br/>';
		}

		$details .= $company->description;

		return $details;
	}

	/**
	 * Prepares the event info to be shown on event details screen
	 *
	 * @param $event object
	 *
	 * @return string
	 *
	 * @since 5.1.0
	 */
	public function prepareEventDetails($event) {
		$details = '';
		if (!empty($event->main_subcategory)) {
			$mainCategory = '';
			foreach ($event->categories as $category) {
				if ($category[0] == $event->main_subcategory) {
					$mainCategory = $category[1];
				}
			}

			$details .= '<b>' . JText::_('LNG_CATEGORY') . '</b>: ';
			$details .= $mainCategory . '<br/><br/>';
		}

		$details .= $event->description;

		return $details;
	}

	/**
	 * Prepares the offer info to be shown on offer details screen
	 *
	 * @param $offer object
	 *
	 * @return string
	 *
	 * @since 5.1.0
	 */
	public function prepareOfferDetails($offer) {
		$details = '';
		if (!empty($offer->main_subcategory)) {
			$mainCategory = '';
			if (!empty($offer->categories)) {
				foreach ($offer->categories as $category) {
					if ($category[0] == $offer->main_subcategory) {
						$mainCategory = $category[1];
					}
				}

				$details .= '<b>' . JText::_('LNG_CATEGORY') . '</b>: ';
				$details .= $mainCategory . '<br/><br/>';
			}
		}

		if (!empty($offer->price)) {
			$details .= '<b>' . JText::_('LNG_PRICE') . '</b>: ';
			if (!empty($offer->specialPrice)) {
				$details .= '<s>' . JBusinessUtil::getPriceFormat($offer->price, $offer->currencyId) . '</s>&nbsp;&nbsp;' . JBusinessUtil::getPriceFormat($offer->specialPrice, $offer->currencyId);
			} else {
				$details .= JBusinessUtil::getPriceFormat($offer->price, $offer->currencyId);
			}
		}

		$details .= $offer->description;

		return $details;
	}

	public function getTrip() {
		$model = $this->getModel('Trip');
		$trip = $model->getTripMobile();

		foreach ($trip->pictures as $picture) {
			$trip->images[] = $picture->picture_path;
		}

		if (count($trip->images) == 0) {
			$trip->images[] = '/no_image.jpg';
		}

		$trip->id       = (int) $trip->id;
		$trip->alias    = isset($trip->alias) ? $trip->alias : '';
		// $trip->dates 	= $model->getTripAvailableDates($trip->id);

		$this->sendResponse($trip);
	}

	public function getTrips() {

		$limitStart = (int) JFactory::getApplication()->input->get("limitstart");
		$limit      = (int) JFactory::getApplication()->input->get("limit");

		if ($limitStart < $limit && $limitStart != 0) {
			$limitStart = $limit;
			JFactory::getApplication()->input->set("limitstart", $limitStart);
		}
		
		$model = $this->getModel('Trips');
		$items = $model->getItemsMobile();

		foreach($items as $item) {
			$item->pictures = explode(',' , $item->pictures);
			$item->logo = $item->pictures[0];
		}

		$this->sendResponse($items);
	}

	public function getGuestDetails() {
        $apiModel = $this->getModel();
        $token 		 = JFactory::getApplication()->input->get("token");

        if ($this->validateToken() && !empty($token)) {
            $user = $apiModel->getUserByToken($token);
        }

		$model = $this->getModel('TripBookingDetails');
		$details = $model->getGuestDetails($user->id);

		$this->sendResponse($details);
	}

	public function updateGuestDetails() {
        $data = JFactory::getApplication()->input;
        $apiModel = $this->getModel();
        $token 		 = $data["token"];

        if ($this->validateToken() && !empty($token)) {
            $user = $apiModel->getUserByToken($token);
        }

		if (empty($user->id)) {
			$result = array(0 => JText::_('LNG_USER_DOESNT_EXIST'));
			$this->sendResponse($result, RESPONSE_STATUS_ERROR);
			return;
		}

		$model = $this->getModel('BillingDetails');
		$result = $model->updateBillingDetails($data);
		
		$this->sendResponse($result, RESPONSE_STATUS_SUCCESS);
	}

	public function checkUserBookCapability() { 
		$token 		 = JFactory::getApplication()->input->get("token");

		$user     = null;
		$apiModel = $this->getModel();
		
		if ($this->validateToken() && !empty($token)) {
			$user = $apiModel->getUserByToken($token);
		}

		$result = array();
		if (empty($user)) { 
			$result[] = array('capability' => 2, 'message' => JText::_('LNG_TRIP_BOOKING_NOT_LOGGED_IN')); 
			$this->sendResponse($result, RESPONSE_STATUS_SUCCESS); 
			return; 
		}

		if(!JBusinessUtil::checkUserBookingCapability($user->id)) {
			$result[] = array('capability' => 3, 'message' => JText::_('LNG_TRIP_BOOKING_NO_ACTIVE_PACKAGE')); 
			$this->sendResponse($result, RESPONSE_STATUS_SUCCESS); 
			return; 
		}

		$result[] = array('capability' => 1, 'message' => JText::_('LNG_TRIP_CAN_BOOK')); 
		$this->sendResponse($result, RESPONSE_STATUS_SUCCESS); 
	}

	public function checkTripDateAvailability() { 
		$token  = JFactory::getApplication()->input->get("token");
		$date 	=  JFactory::getApplication()->input->get("date");

		$user     = null;
		$apiModel = $this->getModel();
		
		if ($this->validateToken() && !empty($token)) {
			$user = $apiModel->getUserByToken($token);
		}

		$date = strtotime($date);
		$formattedDate = date('Y-m-d', $date);

		$result = array();
		if(!JBusinessUtil::checkTripBookingAvailability(998, $formattedDate, null)) {
			$result[] = array("availability" => 2, 'message' => JText::_('LNG_TRIP_BOOKING_NOT_AVAILABLE')); 
			$this->sendResponse($result, RESPONSE_STATUS_SUCCESS); 
			return; 
		}

		$result[] = array("availability" => 1, 'message' => JText::_('LNG_TRIP_CAN_BOOK')); 
		$this->sendResponse($result, RESPONSE_STATUS_SUCCESS); 
	}

	public function createTripBooking() { 
		$data     	 = JFactory::getApplication()->input->getArray(); 
		$data 		 = (object) $data;
		$token 		 = JFactory::getApplication()->input->get("token");

		$user     = null;
		$apiModel = $this->getModel();
		if ($this->validateToken() && !empty($token)) {
			$user = $apiModel->getUserByToken($token);
		}

		$tripId   = $data->tripId; 
		$tripDate = $data->tripDate;  

		$serviceDetails = new stdClass();
		$serviceDetails->tripId =  $data->tripId; 
		$serviceDetails->tripDate =  $data->tripDate; 

		$data->serviceDetails = $serviceDetails;
		
		if(!JBusinessUtil::checkUserBookingCapability($user->id)) {
			$result = array(0 => JText::_('LNG_TRIP_BOOKING_NO_ACTIVE_PACKAGE')); 
			$this->sendResponse($result, RESPONSE_STATUS_ERROR); 
			return; 
		}

		$model = $this->getModel('Trip');
		$trip = $model->getTripMobile();
		
		$trip->dates = array();
		foreach($trip->occurrences as $occurrence) {
			$trip->dates[] = $occurrence->start_date;
		}
		
		if(!in_array($tripDate, $trip->dates)){ 
			$result = array(0 => JText::_('LNG_TRIP_BOOKING_DATE_NOT_AVAILABLE')); 
			$this->sendResponse($result, RESPONSE_STATUS_ERROR); 
			return; 
		}

		$detailsModel = $this->getModel('TripBookingDetails');
		$guestDetails = $detailsModel->getGuestDetails($user->id);

		if(!empty($guestDetails->id)) {
			$data->guestDetails = $guestDetails;
		}

		$bookingService = new TripBookingService();
		$result = $bookingService->saveBooking($data);
 
		$this->sendResponse($result, RESPONSE_STATUS_SUCCESS); 
	} 


	//TODO implement token security mechanism
	public function validateToken() {
		$token = JFactory::getApplication()->input->get("token");

		$model  = $this->getModel();
		$device = $model->getDeviceByToken($token);

		if (!empty($device)) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Get's the data and status, checks if there are any errors, and
	 * prints the data and status as JSON.
	 *
	 * @param array|object $data
	 * @param int          $status
	 *
	 * @since 5.0.0
	 */
	public function sendResponse($data, $status = RESPONSE_STATUS_SUCCESS) {
//		if (!empty(error_get_last())) {
//			$status = RESPONSE_STATUS_ERROR;
//			dump(error_get_last());
//		}

        if (ob_get_length() > 0 ) {
            ob_end_clean();
        }

		$tmp = array($data);
		if (is_array($data)) {
			$tmp = $data;
		}

		$data           = array();
		$data['status'] = $status;

		if ($status == RESPONSE_STATUS_SUCCESS) {
			$data['data'] = $tmp;
		} else {
			$data['error'] = $tmp;
		}

		// Send as JSON
		header("Content-Type: application/json", true);
		echo json_encode($data, JSON_PRETTY_PRINT);
		exit;
	}

	public function sendSubscriptionEmail() {
		$data  = JFactory::getApplication()->input->json->getArray();
		$response = EmailService::sendSubscriptionEmail($data);

		if($response) {
			$status = RESPONSE_STATUS_SUCCESS;
		} else {
			$status = RESPONSE_STATUS_ERROR;
		}

		$this->sendResponse($response, $status);
	}
}