<?php
/**
 * @package    JBusinessDirectory
 *
 * @author     CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2021 CMSJunkie. All rights reserved.
 * @license    https://www.gnu.org/licenses/agpl-3.0.en.html
 */
defined('_JEXEC') or die('Restricted access');
JTable::addIncludePath(JPATH_ROOT . '/administrator/components/com_jbusinessdirectory/tables');

class NotificationService {
	private static $url = 'https://fcm.googleapis.com/fcm/send';

	/**
	 * Retrieves the owner of the company that has been bookmarked,
	 * and sends a notification to his mobile device
	 *
	 * @param $itemId int ID of the company/offer
	 * @param $itemType int type of the item (company or offer)
	 *
	 * @return bool
	 * @since 5.0.0
	 */
	public static function sendBookmarkNotification($itemId, $itemType = ITEM_TYPE_BUSINESS) {
		$userId = null;
		$name = '';
		$title = '';
		if ($itemType == ITEM_TYPE_BUSINESS) {
			$companyTable = JTable::getInstance('Company', 'JTable');
			$companyTable->load($itemId);

			$userId = $companyTable->userId;
			if (empty($userId)) {
				return false;
			}

			$title = JText::_('LNG_NEW_COMPANY_BOOKMARK');
			$name = $companyTable->name;
		} else {
			$offerTable = JTable::getInstance('Offer', 'JTable');
			$offerTable->load($itemId);

			$userId = isset($offerTable->user_id) ? $offerTable->user_id : null;
			if (empty($userId)) {
				$companyId = isset($offerTable->companyId) ? $offerTable->companyId : null;
				if (empty($companyId)) {
					return false;
				}

				$companyTable = JTable::getInstance('Company', 'JTable');
				$companyTable->load($companyId);

				$userId = $companyTable->userId;
				if (empty($userId)) {
					return false;
				}
			}

			$title = JText::_('LNG_NEW_OFFER_BOOKMARK');
			$name = $offerTable->subject;
		}

		$deviceTable = JTable::getInstance('MobileDevice', 'JTable');
		$devices     = $deviceTable->getDevicesByUser($userId);

		$body = '';
		$body .= $name . ': ';
		$body .= $title;

		$notification             = array();
		$notification["title"]    = JText::_('LNG_NEW_BOOKMARK');
		$notification["body"]     = $body;
		$notification["type"]     = NOTIFICATION_TYPE_BOOKMARK;
		$notification["itemId"]   = (int) $itemId;
		$notification["itemType"] = $itemType;

		$receivers = array();
		foreach ($devices as $device) {
			$receivers[] = $device->firebase_token;
		}

		if (count($receivers) == 0) {
			return false;
		}

		return self::sendNotification($notification, $receivers);
	}

	/**
	 * Retrieves the owner of the company that has been reviewed,
	 * and sends a notification to his mobile device
	 *
	 * @param $itemId int ID of the company/offer
	 * @param $itemType int type of the item (company or offer)
	 *
	 * @return bool
	 * @since 5.0.0
	 */
	public static function sendReviewNotification($itemId, $itemType = ITEM_TYPE_OFFER) {
		$userId = null;
		$name = '';
		$title = '';
		if ($itemType == ITEM_TYPE_BUSINESS) {
			$companyTable = JTable::getInstance('Company', 'JTable');
			$companyTable->load($itemId);

			$userId = $companyTable->userId;
			if (empty($userId)) {
				return false;
			}

			$title = JText::_('LNG_NEW_COMPANY_REVIEW');
			$name = $companyTable->name;
		} else {
			$offerTable = JTable::getInstance('Offer', 'JTable');
			$offerTable->load($itemId);

			$userId = isset($offerTable->user_id) ? $offerTable->user_id : null;
			if (empty($userId)) {
				$companyId = isset($offerTable->companyId) ? $offerTable->companyId : null;
				if (empty($companyId)) {
					return false;
				}

				$companyTable = JTable::getInstance('Company', 'JTable');
				$companyTable->load($companyId);

				$userId = $companyTable->userId;
				if (empty($userId)) {
					return false;
				}
			}

			$title = JText::_('LNG_NEW_OFFER_REVIEW');
			$name = $offerTable->subject;
		}

		$deviceTable = JTable::getInstance('MobileDevice', 'JTable');
		$devices     = $deviceTable->getDevicesByUser($userId);

		$body = '';
		$body .= $name . ': ';
		$body .= $title;

		$notification             = array();
		$notification["title"]    = JText::_('LNG_NEW_REVIEW');
		$notification["body"]     = $body;
		$notification["type"]     = NOTIFICATION_TYPE_REVIEW;
		$notification["itemId"]   = (int) $itemId;
		$notification["itemType"] = $itemType;

		$receivers = array();
		foreach ($devices as $device) {
			$receivers[] = $device->firebase_token;
		}

		if (count($receivers) == 0) {
			return false;
		}

		return self::sendNotification($notification, $receivers);
	}

	/**
	 * Sends a notification to the receiver's mobile device
	 *
	 * @param $senderId   int ID of the sender
	 * @param $receiverId int ID of the receiver
	 * @param $subject    string message subject
	 * @param $body       string message body
	 *
	 * @return bool
	 * @since 5.0.0
	 */
	public static function sendMessageNotification($senderId, $receiverId, $subject, $body) {
		$deviceTable = JTable::getInstance('MobileDevice', 'JTable');
		$devices     = $deviceTable->getDevicesByUser($receiverId);

		$notification                 = array();
		$notification["title"]        = $subject;
		$notification["body"]         = $body;
		$notification["type"]         = NOTIFICATION_TYPE_MESSAGE;

		$notification["senderId"]   = $senderId;

		$receivers = array();
		foreach ($devices as $device) {
			$receivers[] = $device->firebase_token;
		}

		if (count($receivers) == 0) {
			return false;
		}

		return self::sendNotification($notification, $receivers);
	}

	/**
	 * Sends the notification to the specified receivers.
	 *
	 * @param $notification array data for the notification
	 * @param $receivers array firebase tokens of the receivers
	 *
	 * @return bool
	 * @since 5.0.0
	 */
	private static function sendNotification($notification, $receivers) {
		$applicationSettings = JBusinessUtil::getApplicationSettings();

		$fields = array(
			'registration_ids' => $receivers,
			'data'         => array(
				'title' => $notification['title'],
				'body'  => $notification['body'],
				"type"     => $notification['type'],
				"target"   => "test",
				"itemId"   => $notification['itemId'],
				"itemType" => $notification['itemType'],
				"timestamp" => date("Y-m-d G:i:s")
			)
		);

		if ($notification['type'] == NOTIFICATION_TYPE_MESSAGE) {
			$fields = array(
				'registration_ids' => $receivers,
				'data'             => array(
					'title'     => $notification['title'],
					'body'      => $notification['body'],
					"type"      => $notification['type'],
					"target"    => "test",
					"id"        => $notification['senderId'],
					"name"      => $notification['title'],
					"timestamp" => date("Y-m-d G:i:s")
				)
			);
		}

		$headers = array(
			'Authorization: key='."$applicationSettings->firebase_server_key",
			'Content-Type: application/json'
		);

		$fields = json_encode($fields);

		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, self::$url);
		curl_setopt($ch, CURLOPT_POST, true);
		curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $fields);

		$result = curl_exec($ch);
		curl_close($ch);

		$result = json_decode($result);
		if ($result->failure) {
			return false;
		}

		return true;
	}

	/**
	 * Sends a notification to all users that have subscribed themselves to a specific topic.
	 *
	 * @param $notification array data for the notification
	 * @param $topic string name of the topic
	 *
	 * @return bool
	 * @since 5.0.0
	 */
	public static function sendTopicNotifications($notification, $topic) {
		$applicationSettings = JBusinessUtil::getApplicationSettings();

		$fields = array(
			'notification'     => array(
				'title' => $notification['title'],
				'body'  => $notification['body'],
			),
			'to' => '/topics/'.$topic
		);

		$headers = array(
			'Authorization: key='."$applicationSettings->firebase_server_key",
			'Content-Type: application/json'
		);

		$fields = json_encode($fields);

		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, self::$url);
		curl_setopt($ch, CURLOPT_POST, true);
		curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $fields);

		$result = curl_exec($ch);
		curl_close($ch);

		$result = json_decode($result);
		if (!isset($result->message_id)) {
			return false;
		}

		return true;
	}
}
