<?php
/**
 * @package    JBusinessDirectory
 *
 * @author     CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2021 CMSJunkie. All rights reserved.
 * @license    https://www.gnu.org/licenses/agpl-3.0.en.html
 */
defined('_JEXEC') or die('Restricted access');

class CampaignUserDataService {
	/**
	 * Create user data for storing booking details
	 *
	 * @param $data
	 * @param $userData
	 *
	 * @return stdClass
	 *
	 * @since 5.2.0
	 */
	public static function createUserData($data, $userData) {
		if (empty($userData)) {
			$userData = new stdClass();
		}

		if (!isset($userData->first_name)) {
			$userData->first_name = '';
		}
		if (!isset($userData->last_name)) {
			$userData->last_name = '';
		}
		if (!isset($userData->address)) {
			$userData->address = '';
		}
		if (!isset($userData->city)) {
			$userData->city = '';
		}
		if (!isset($userData->state_name)) {
			$userData->state_name = '';
		}
		if (!isset($userData->country)) {
			$userData->country = '';
		}
		if (!isset($userData->postal_code)) {
			$userData->postal_code = '';
		}
		if (!isset($userData->phone)) {
			$userData->phone = '';
		}
		if (!isset($userData->email)) {
			$userData->email = '';
		}
		if (!isset($userData->conf_email)) {
			$userData->conf_email = '';
		}

		return $userData;
	}

	/**
	 * Initialize user data
	 *
	 * @since 5.2.0
	 */
	public static function initializeUserData($resetUserData = false) {
		$get  = JFactory::getApplication()->input->get->getArray();
		$data = JFactory::getApplication()->input->post->getArray();
		if (count($data) == 0) {
			$data = $get;
		}

		$userData = isset($_SESSION['userData']) ? $_SESSION['userData'] : null;
		if (!isset($userData) || $resetUserData) {
			$userData             = self::createUserData($data, $userData);
			$_SESSION['userData'] = $userData;
		}

		if (!isset($userData->guestDetails)) {
			$guestDtls               = new stdClass();
			$guestDtls->first_name   = "";
			$guestDtls->last_name    = "";
			$guestDtls->address      = "";
			$guestDtls->city         = "";
			$guestDtls->county       = "";
			$guestDtls->country_name = "";
			$guestDtls->postalCode   = "";
			$guestDtls->phone        = "";
			$guestDtls->email        = "";
			$userData->guestDetails  = $guestDtls;
		}

		$_SESSION['userData'] = $userData;
		return $userData;
	}

	/**
	 * Get user data object created from session data
	 * @return mixed|null|stdClass
	 *
	 * @since 5.2.0
	 */
	public static function getUserData() {
		$session  = self::getJoomlaSession();
		$userData = isset($_SESSION['userData']) ? $_SESSION['userData'] : null;
		if (!isset($userData)) {
			$userData             = self::initializeUserData();
			$_SESSION['userData'] = $userData;
		}
		if (empty($userData->eventId)) {
			$userData->eventId    = 0;
			$_SESSION['userData'] = $userData;
		}

		return $userData;
	}


	/**
	 * @param $guestDetails array
	 *
	 * @since 5.2.0
	 */
	public static function addGuestDetails($guestDetails) {
		$userData                = $_SESSION['userData'];
		$guestDtls               = new stdClass();
		$guestDtls->first_name   = ucfirst($guestDetails["first_name"]);
		$guestDtls->last_name    = ucfirst($guestDetails["last_name"]);
		$guestDtls->address      = ucfirst($guestDetails["address"]);
		$guestDtls->city         = $guestDetails["city"];
		$guestDtls->county       = $guestDetails["region"];

		if(!is_object($guestDetails["country"])){
			$guestDetails["country"] = JBusinessUtil::getCountry($guestDetails["country"]);
		}

		$guestDtls->country = $guestDetails["country"];
		$guestDtls->country_name = !empty($guestDetails["country"])?$guestDetails["country"]->country_name:"";

		$guestDtls->country      = $guestDetails["country"];
		$guestDtls->postalCode   = strtoupper($guestDetails["postal_code"]);
		$guestDtls->phone        = $guestDetails["phone"];
		$guestDtls->email        = $guestDetails["email"];

		$userData->guestDetails = $guestDtls;

		$_SESSION['userData'] = $userData;
	}

	/**
	 * @param $currencyName
	 * @param $currencySymbol
	 *
	 * @since 5.2.0
	 */
	public static function setCurrency($currencyName, $currencySymbol) {
		$currency         = new stdClass();
		$currency->name   = $currencyName;
		$currency->symbol = $currencySymbol;

		$session            = self::getJoomlaSession();
		$userData           = $_SESSION['userData'];
		$userData->currency = $currency;

		if ($userData->user_currency == "") {
			$userData->user_currency = $currency->name;
		}
		$_SESSION['userData'] = $userData;
	}

	private static function getJoomlaSession() {
		$session = JFactory::getSession();
		if ($session->getState() !== 'active') {
			$app = JFactory::getApplication();
			$msg = "Your session has expired";
			$app->redirect('index.php?option=' . JBusinessUtil::getComponentName() . '&view=events', $msg);
			$app->enqueueMessage("Your session has expired", 'warning');
		} else {
			return $session;
		}
	}

	/**
	 * Create the guest details summary
	 *
	 * @param $guestDetails
	 *
	 * @return string
	 * @since 5.0.0
	 */
	public static function getGuestDetailsSummary($guestDetails) {
		$result = "";
		$result .= "<div class=\"booking-guest-details\">";
		$result .= "<h4>" . JText::_("LNG_BILLING_DETAILS") . "</h4>";
		$result .= "<div class=\"row\">";
		$result .= "<div class=\"col-md-7\">";
		$result .= $guestDetails->first_name . " " . $guestDetails->last_name;
		$result .= "</div>";
		$result .= "</div>";

		$result .= "<div class=\"row\">";
		$result .= "<div class=\"col-md-7\">";
		$result .= JBusinessUtil::getAddressText($guestDetails);
		$result .= "</div>";
		$result .= "</div>";

		$result .= "<div class=\"row\">";
		$result .= "<div class=\"col-md-7\">";
		$result .= $guestDetails->email;
		$result .= "</div>";
		$result .= "</div>";

		$result .= "<div class=\"row\">";
		$result .= "<div class=\"col-md-7\">";
		$result .= $guestDetails->phone;
		$result .= "</div>";
		$result .= "</div>";

		$result .= "</div>";

		return $result;
	}

	/**
	 * Creates the campaign html summary
	 *
	 * @param $campaign object
	 * @return string
	 *
	 * @since 5.2.0
	 */
	public static function getCampaignSummary($campaign) {
		$result = "";
		$result .= "<div class=\"booking-guest-details\">";
		$result .= "<h4>" . JText::_("LNG_CAMPAIGN_DETAILS") . "</h4>";
		$result .= "<div class=\"row\">";
		$result .= "<div class=\"col-md-5\">";
		$result .= "<strong>" . JText::_("LNG_NAME") . "</strong>";
		$result .= "</div>";
		$result .= "<div class=\"col-md-7\">";
		$result .= $campaign->name;
		$result .= "</div>";
		$result .= "</div>";

		$result .= "<div class=\"row\">";
		$result .= "<div class=\"col-md-5\">";
		$result .= "<strong>" . JText::_("LNG_COMPANY") . "</strong>";
		$result .= "</div>";
		$result .= "<div class=\"col-md-7\">";
		$result .= $campaign->company;
		$result .= "</div>";
		$result .= "</div>";

		$result .= "<div class=\"row\">";
		$result .= "<div class=\"col-md-5\">";
		$result .= "<strong>" . JText::_("LNG_BUDGET") . "</strong>";
		$result .= "</div>";
		$result .= "<div class=\"col-md-7\">";
		$result .= JBusinessUtil::getPriceFormat($campaign->budget);
		$result .= "</div>";
		$result .= "</div>";

		//applying taxes on order
		$basePrice = $campaign->budget;
		$initialAmount = $basePrice;
		$orderTable = JTable::getInstance("Order", "JTable", array());
		$taxes = $orderTable->getOrderTaxes($campaign->id, JBD_APP_CAMPAIGNS);
		if (!empty($taxes)) {
			foreach ($taxes as &$tax) {
				if ($tax->tax_type==1) {
					$result .= "<div class=\"row\">";
					$result .= "<div class=\"col-md-5\">";
					$result .= "<strong>" . $tax->tax_name . "</strong>";
					$result .= "</div>";
					$result .= "<div class=\"col-md-7\">";
					$result .= JBusinessUtil::getPriceFormat($tax->tax_amount);
					$result .= "</div>";
					$result .= "</div>";
					$basePrice += $tax->tax_amount;
				} else {
					$taxAmount = $tax->tax_amount*$initialAmount/100;
					$result .= "<div class=\"row\">";
					$result .= "<div class=\"col-md-5\">";
					$result .= "<strong>" . $tax->tax_name." ". "(".$tax->tax_amount."%)" . "</strong>";
					$result .= "</div>";
					$result .= "<div class=\"col-md-7\">";
					$result .= JBusinessUtil::getPriceFormat($taxAmount);
					$result .= "</div>";
					$result .= "</div>";
					$basePrice += $taxAmount;
				}
			}
		}

		$result .= "<hr/>";
		$result .= "<div class=\"row\">";
		$result .= "<div class=\"col-md-5\">";
		$result .= "<strong>" . JText::_("LNG_TOTAL") . "</strong>";
		$result .= "</div>";
		$result .= "<div class=\"col-md-7\">";
		$result .= JBusinessUtil::getPriceFormat($basePrice);
		$result .= "</div>";
		$result .= "</div>";

		$result .= "</div>";

		return $result;
	}
}
