<?php
/**
 * @package    JBusinessDirectory
 *
 * @author CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2021 CMSJunkie. All rights reserved.
 * @license   https://www.gnu.org/licenses/agpl-3.0.en.html
 */
defined('_JEXEC') or die('Restricted access');
use Joomla\Utilities\ArrayHelper;
require_once(HELPERS_PATH.'/category_lib.php');
require_once BD_CLASSES_PATH.'/attributes/attributeservice.php';


jimport('joomla.application.component.modeladmin');
/**
 * Video Model for Videos.
 *
 */
class JBusinessDirectoryModelVideo extends JModelAdmin {


	/**
	 * @var        string    The prefix to use with controller messages.
	 * @since   1.6
	 */
	protected $text_prefix = 'COM_JBUSINESSDIRECTORY_VIDEO';
	protected $mainSubcategory = 0;

	/**
	 * Model context string.
	 *
	 * @var        string
	 */
	protected $_context = 'com_jbusinessdirectory.video';

	public function __construct($config = array()) {
		$this->appSettings = JBusinessUtil::getApplicationSettings();
	
		parent::__construct($config);
	}
	
	/**
	 * Method to test whether a record can be deleted.
	 *
	 * @param   object    A record object.
	 *
	 * @return  boolean  True if allowed to delete the record. Defaults to the permission set in the component.
	 */
	protected function canDelete($record) {
		return true;
	}

	/**
	 * Method to test whether a record can be deleted.
	 *
	 * @param   object    A record object.
	 *
	 * @return  boolean  True if allowed to change the state of the record. Defaults to the permission set in the component.
	 */
	protected function canEditState($record) {
		return true;
	}

	/**
	 * Returns a Table object, always creating it
	 *
	 * @param   type    The table type to instantiate
	 * @param   string    A prefix for the table class name. Optional.
	 * @param   array  Configuration array for model. Optional.
	 * @return  JTable    A database object
	 */
	public function getTable($type = 'Videos', $prefix = 'Table', $config = array()) {
		return JTable::getInstance($type, $prefix, $config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @since   1.6
	 */
	protected function populateState() {
		$app = JFactory::getApplication('administrator');

		// Load the User state.
		$id = JFactory::getApplication()->input->getInt('id');
		$this->setState('video.id', $id);
	}

	/**
	 * Method to get a menu item.
	 *
	 * @param   integer    The id of the menu item to get.
	 *
	 * @return  mixed  Menu item data object on success, false on failure.
	 */
	public function &getItem($itemId = null) {
		$itemId = (!empty($itemId)) ? $itemId : (int)$this->getState('video.id');
		$false = false;

		// Get a menu item row instance.
		$table = $this->getTable();

		// Attempt to load the row.
		$return = $table->load($itemId);

		// Check for a table object error.
		if ($return === false && $table->getError()) {
			$this->setError($table->getError());
			return $false;
		}

		$properties = $table->getProperties(1);
		$value = ArrayHelper::toObject($properties, 'JObject');

		$companyCategoryTable = $this->getTable('CompanyCategory', 'JTable');
		if (!empty($itemId)) {
			$value->selCats = $companyCategoryTable->getSelectedVideoCategoriesList($itemId);
		} else {
			$value->selCats = array();
		}

		$value->selectedCategories = $companyCategoryTable->getSelectedVideoCategories($itemId);
		foreach ($value->selectedCategories as $cat) {
			$cat->name = str_repeat('- ', $cat->level - 1) . $cat->name;
		}
		$value->defaultAtrributes = JBusinessUtil::getAttributeConfiguration(DEFAULT_ATTRIBUTE_TYPE_VIDEO);

		$categoryId = $this->appSettings->enable_attribute_category?$value->main_subcategory:null;
		$attributesTable = $this->getTable('VideoAttributes', "JTable");
		$value->customFields = $attributesTable->getVideoAttributes($itemId, $categoryId);

		if ($this->appSettings->enable_multilingual) {
			JBusinessDirectoryTranslations::updateAttributesTranslation($value->customFields);
		}

		return $value;
	}

	/**
	 * Method to get the menu item form.
	 *
	 * @param   array  $data		Data for the form.
	 * @param   boolean	$loadData	True if the form is to load its own data (default case), false if not.
	 * @return  JForm	A JForm object on success, false on failure
	 * @since   1.6
	 */
	public function getForm($data = array(), $loadData = true) {
		exit;
		// The folder and element vars are passed when saving the form.
		if (empty($data)) {
			$item		= $this->getItem();
			// The type should already be set.
		}
		// Get the form.
		$form = $this->loadForm('com_jbusinessdirectory.video', 'item', array('control' => 'jform', 'load_data' => $loadData), true);
		if (empty($form)) {
			return false;
		}

		return $form;
	}

	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return  mixed  The data for the form.
	 * @since   1.6
	 */
	protected function loadFormData() {
		// Check the session for previously entered form data.
		$data = JFactory::getApplication()->getUserState('com_jbusinessdirectory.edit.video.data', array());

		if (empty($data)) {
			$data = $this->getItem();
		}

		return $data;
	}


	/**
	 * Check for duplicate alias and generate a new alias
	 * @param unknown_type $busienssId
	 * @param unknown_type $alias
	 */
	public function checkAlias($videoId, $alias) {
		$table = $this->getTable();
		while ($table->checkAlias($videoId, $alias)) {
			$alias = Joomla\String\StringHelper::increment($alias, 'dash');
		}
		return $alias;
	}

	/**
	 * Method to save the form data.
	 *
	 * @param   array  The form data.
	 * @return  boolean  True on success.
	 */

	public function save($data) {
		$id	= (!empty($data['id'])) ? $data['id'] : (int) $this->getState('video.id');
		if (empty($data['id'])) {
			$data['id'] = 0;
		}
		$isNew = true;

		$input = JFactory::getApplication()->input;
		$video_description = $input->get("description", '', 'RAW');
		$transcript = $input->get("transcript", '', 'RAW');
		
		$data["alias"]= JBusinessUtil::getAlias($data["name"], $data["alias"]);
		$data["alias"] = $this->checkAlias($id, $data["alias"]);

		if (!empty($video_description)) {
			$data["description"] = $video_description;
		}

		if (!empty($transcript)) {
			$data["transcript"] = $transcript;
		}
		
		// delete main category if not present in data
		if (!isset($data["main_subcategory"])) {
			$data["main_subcategory"] = 0;
		}
		
		// Get a row instance.
		$table = $this->getTable();

		// Load the row if saving an existing item.
		if ($id > 0) {
			$table->load($id);
			$isNew = false;
		}

		// Bind the data.
		if (!$table->bind($data)) {
			$this->setError($table->getError());
			return false;
		}

		// Check the data.
		if (!$table->check()) {
			$this->setError($table->getError());
			return false;
		}

		// Store the data.
		if (!$table->store()) {
			$this->setError($table->getError());
			return false;
		}

		$this->setState('video.id', $table->id);
		
		try {
			$this->storeAttributes($this->getState('video.id'), $data);
		} catch (Exception $ex) {
			$this->setError($ex->getMessage());
		}


		// if no category is selected, create a dummy relation with categoryId = -1 so that
		// the insertVideoRelations function deletes all other existing relations
		if (!isset($data['selectedSubcategories'])) {
			$data['selectedSubcategories'] = array(-1);
		}

		//save in video_category table
		$table = $this->getTable('CompanyCategory', "JTable");
		if (!empty($data["selectedSubcategories"])) {
			$table->insertVideoRelations($this->getState('video.id'), $data["selectedSubcategories"]);
		}


		// Clean the cache
		$this->cleanCache();

		return true;
	}

	/**
	 * Method to delete groups.
	 *
	 * @param   array  An array of item ids.
	 * @return  boolean  Returns true on success, false on failure.
	 */
	public function delete(&$itemIds) {
		// Sanitize the ids.
		$itemIds = (array) $itemIds;
		ArrayHelper::toInteger($itemIds);

		// Get a group row instance.
		$table = $this->getTable();

		// Iterate the items to delete each one.
		foreach ($itemIds as $itemId) {
			if (!$table->delete($itemId)) {
				$this->setError($table->getError());
				return false;
			}
		}

		// Clean the cache
		$this->cleanCache();

		return true;
	}

	/**
	 * Method that retrieves custom attributes based on a certain category in order to be
	 * rendered in HTML.
	 * If multilingual is also enabled, it will translate the attributes.
	 *
	 * @param $categoryId int ID of the category
	 * @param $videoId int ID of the video
	 *
	 * @return string html output
	 */
	public function getAttributesAjax($categoryId, $videoId) {
		$attributesTable = $this->getTable('Attribute', 'JTable');
		$customFields = $attributesTable->getAttributesByCategory($categoryId, 4, $videoId);

		foreach ($customFields as $val) {
			if (!isset($val->attributeValue)) {
				$val->attributeValue = '';
			}
		}

		if (!empty($customFields)) {
			if ($this->appSettings->enable_multilingual) {
				JBusinessDirectoryTranslations::updateAttributesTranslation($customFields);
			}

			$renderedContent = AttributeService::renderAttributes($customFields, false, array());
		} else {
			$renderedContent = null;
		}

		return $renderedContent;
	}

	public function getCategories() {
		$categoryService = new JBusinessDirectorCategoryLib();
		$categoryTable = $this->getTable("Category", "JBusinessTable");
		$categories = $categoryTable->getAllCategories(CATEGORY_TYPE_VIDEO);
		$categories = $categoryService->processCategoriesByName($categories);
		return $categories;
	}

	public function getCategoryByName($categories, $categoryName) {
		$categoryService = new JBusinessDirectorCategoryLib();
		$cat = null;
		$category = $categoryService->findCategoryByName($categories, $cat, $categoryName);

		return $category;
	}

	public function storeAttributes($videoId, $data) {
		#delete all ad attributes
		
		$attrTable =$this->getTable('VideoAttributes', 'JTable');
		$attrIds=array();
		foreach ($data as $key => $value) {
			#save ad attributes
			if (strpos($key, "attribute")===0) {
				$attributeArr = explode("_", $key);
				if (!empty($attributeArr[1])) {
					$attrIds[]=$attributeArr[1];
				}
			}
			if (strpos($key, "delete_attribute")===0) {
				$attributeArr = explode("_", $key);
				if (!empty($attributeArr[2])) {
					$attrIds[]=$attributeArr[2];
				}
			}
		}
		
		$attrIds = array_unique($attrIds);

		if (!empty($attrIds) && !$attrTable->deleteVideoAttributes($videoId, $attrIds)) {
			$this->setError(JText::_("LNG_ERROR_DELETING_AD_ATTRIBUTES").$attrTable->getError());
		}
		
		foreach ($data as $key => $value) {
			#save ad attributes
			if (strpos($key, "attribute")===0) {
				$attributeArr = explode("_", $key);
				//print_r($attributeArr);
				$videoAttributeTable =$this->getTable('VideoAttributes', 'JTable');
				$videoAttributeTable->video_id= $videoId;
				if(is_int($value)){
					$videoAttributeTable->option_id= $value;
				}
				$videoAttributeTable->value= $value;
				$videoAttributeTable->attribute_id= $attributeArr[1];

				if (is_array($videoAttributeTable->value)) {
					$videoAttributeTable->value = implode(",", $videoAttributeTable->value);
				}

				$properties = $videoAttributeTable->getProperties(1);
				$value = ArrayHelper::toObject($properties, 'JObject');

				if (!$videoAttributeTable->store()) {
					$this->setError(JText::_("LNG_ERROR_SAVING_AD_ATTRIBUTES").$videoAttributeTable->getError());
				}
			}
		}
	}

	public function importVideosAndSetMessage($filePath, $delimiter, $videoArray) {
		$result = $this->importVideos($filePath, $delimiter, $videoArray);

		$message = "";

		if ($result->differences) {
			$message = $message . $message = $message . JFactory::getApplication()->enqueueMessage(JText::_('LNG_NOT_ALLOWED_HEADER'), 'warning');
			foreach ($result->differences as $header) {
				$message = $message . JFactory::getApplication()->enqueueMessage($header, 'warning');
			}
			$message = $message . JFactory::getApplication()->enqueueMessage("<br/>", 'warning');
			$message = $message . $message = $message . JFactory::getApplication()->enqueueMessage(JText::_('LNG_ALLOWED_HEADER'), 'warning');
			foreach ($result->correctHeader as $header) {
				$message = $message . JFactory::getApplication()->enqueueMessage($header, 'warning');
			}
			$message = $message . JFactory::getApplication()->enqueueMessage("<br/>", 'warning');
		}

		if ($result->failedOffers) {
			$message = $message . JFactory::getApplication()->enqueueMessage(JText::_('LNG_IMPORT_FAILED_FOR'), 'warning');
			foreach ($result->failedOffers as $failed) {
				$message = $message . JFactory::getApplication()->enqueueMessage(JText::_('LNG_ROW') . " " . $failed->row . "  " . JText::_('LNG_VIDEO_NAME') . " " . $failed->name, 'warning');
			}
			$message = $message . JFactory::getApplication()->enqueueMessage("<br/>", 'warning');
		}

		$message = $message . JText::plural('COM_JBUSINESS_DIRECTORY_N_VIDEOS_IMPORTED', $result->newVideos);
		$message = $message . "<br/>";

		return $message;
	}

	public function importVideos ($filePath, $delimiter, $txtInfo) {
		$tableheader = array("name", "url", "description", "main_subcategory","duration", "transcript");

		$newVideosCount = 0;
		$nrHeader = count($tableheader);
		$failedVideos = array();
		$error_row = 0;
		//        $updateExisting = JFactory::getApplication()->input->get("update_existing");
		ini_set("auto_detect_line_endings", "1");
		$count = 0;
		$row = 1;
		if ((($handle = fopen($filePath, "r")) !== false) || ($txtInfo != null)) {
			//if there are data on the text area the header will be filled here
			if ($txtInfo != null) {
				for ($i = 0; $i < $nrHeader; $i++) {
					$header[$i] = $txtInfo[$i];
					$count++;
				}
			}

			$data = null;

			if ($txtInfo != null) {
				$maxElementOnArea = count($txtInfo);
			} else {
				$maxElementOnArea = 2;
			}
			while (($data = fgetcsv($handle, 9000, $delimiter)) !== false && ($maxElementOnArea - $count) > 0) {
				$video = array();

				//if there are no data on text area then we will need to fill here the header
				if ($txtInfo == null) {
					if ($row == 1) {
						$header = $data;
						$row++;
						continue;
					}
				} else {
					//else if there are data then we will need to fill the data array every time inside the while with the data
					$data = null;
					$datacount = 0;
					for ($i = $count; $i < $count + $nrHeader; $i++) {
						$data[$datacount] = $txtInfo[$i];
						$datacount++;
					}
				}

				//this function help the while when it has importing from a file.. it help to make the while function finish if there is no more data
				if ($txtInfo == null) {
					if ($data === false) {
						break;
					}
				}

				//check if the header are OK
				$header_differences = array_diff($header, $tableheader);
				if ($header_differences != null) {
					break;
				}

				$num = count($data);
				//dump($data);
				//echo "<p> $num fields in line $row: <br /></p>\n";
				$row++;
				for ($c = 0; $c < $num; $c++) {
					$video[strtolower($header[$c])] = $data[$c];
				}

				$datas = array();			
				
				$datas["name"] = isset($video["name"]) ? $video["name"] : "";
				$datas["url"] = !empty($video["url"]) ? $video["url"] : "";
				$datas["description"] = isset($video["description"]) ? $video["description"] : "";
				$datas["main_subcategory"] = isset($video["main_subcategory"]) ? $video["main_subcategory"] : "";
				$datas["duration"] = isset($video["duration"]) ? $video["duration"] : "";
				$datas["transcript"] = !empty($video["transcript"]) ? $video["transcript"] : "";				
				
				try {
					$this->setState('video.id', 0);
					$error_row++;
					if ($this->save($datas)) {
						$newVideosCount++;
					} else {
						$failedVideo = new stdClass();
						$failedVideo->row = $error_row;
						array_push($failedVideos, $failedVideo);
					}
				} catch (Exception $e) {
					dump($e);
				}
				if ($txtInfo != null) {
					$count += $nrHeader;
				}
				
			}
			fclose($handle);
		}

		$result = new stdClass();
		$result->differences = $header_differences;
		$result->correctHeader = $tableheader;
		$result->failedVideos = $failedVideos;
		$result->newVideos = $newVideosCount;

		return $result;
	}
}
