<?php
/**
 * @package    JBusinessDirectory
 *
 * @author     CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2021 CMSJunkie. All rights reserved.
 * @license    https://www.gnu.org/licenses/agpl-3.0.en.html
 */

defined('_JEXEC') or die('Restricted access');
jimport('joomla.application.component.modellist');

/**
 * Packages List Model
 *
 * @package     JBusinessDirectory
 * @subpackage  com_jbusinessdirectory
 */
class JBusinessDirectoryModelPackages extends JModelList {
	/**
	 * Constructor.
	 *
	 * @param   array  An optional associative array of configuration settings.
	 *
	 * @see     JController
	 * @since   1.6
	 */
	public function __construct($config = array()) {
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = array(
				'id', 'p.id',
				'name', 'p.name',
				'price', 'p.price',
				"special_price", "p.special_price",
				"days", "p.days",
				"special_from_date", "p.special_from_date",
				"special_to_date", "p.special_to_date",
				"status", "p.status",
				'ordering', 'p.ordering',
				"popular", "p.popular",
				'package_type', 'p.package_type',
			);
		}

		parent::__construct($config);
	}

	/**
	 * Overrides the getItems method to attach additional metrics to the list.
	 *
	 * @return  mixed  An array of data items on success, false on failure.
	 *
	 * @since   1.6.1
	 */
	public function getItems() {
		// Get a storage key.
		$store = $this->getStoreId('getItems');

		// Try to load the data from internal storage.
		if (!empty($this->cache[$store])) {
			return $this->cache[$store];
		}

		// Load the list items.
		$items = parent::getItems();

		// If empty or an error, just return.
		if (empty($items)) {
			return array();
		}

		$packagesTable   = JTable::getInstance('Package');
		$packageFeatures = array_merge(JBusinessDirectoryHelper::getPackageFeatures(), JBusinessDirectoryHelper::getUserPackageFeatures());

		foreach ($items as $item) {
			$item->special_from_date = JBusinessUtil::convertToFormat($item->special_from_date);
			$item->special_to_date   = JBusinessUtil::convertToFormat($item->special_to_date);

			//add the features to the item
			if (!empty($item->features)) {
				$item->features = explode(",", $item->features);
				$features       = array();
				foreach ($item->features as $feature) {
					if (!empty($packageFeatures[$feature])) {
						$features[] = $packageFeatures[$feature];
					} else {
						$features[] = $feature;
					}
				}
				$item->features = implode(', ', $features);
			}
		}

		// Add the items to the internal cache.
		$this->cache[$store] = $items;

		return $this->cache[$store];
	}

	/**
	 * Method to build an SQL query to load the list data.
	 *
	 * @return  string  An SQL query
	 *
	 * @since   1.6
	 */
	protected function getListQuery() {
		// Create a new query object.
		$db    = $this->getDbo();
		$query = $db->getQuery(true);

		// Select all fields from the table.
		$query->select($this->getState('list.select', 'p.*'));
		$query->from($db->quoteName('#__jbusinessdirectory_packages') . ' AS p');

		// Join over the company types
		$query->select('GROUP_CONCAT(pf.feature) as features');
		$query->join('LEFT', $db->quoteName('#__jbusinessdirectory_package_fields') . ' AS pf ON pf.package_id=p.id');

		$packageType = $this->getState('filter.package_type');
		if (is_numeric($packageType) && $packageType != 0) {
			$query->where('p.package_type ='.(int) $packageType);
		}

		// Filter by search in title.
		$search = $this->getState('filter.search');
		if (!empty($search)) {
			$query->where("(p.name LIKE '%" . trim($db->escape($search)) . "%' or p.description LIKE '%" . trim($db->escape($search)) . "%')");
		}

		$statusId = $this->getState('filter.status_id');
		if (is_numeric($statusId)) {
			$query->where("p.status =" . (int) $statusId);
		}

		$query->group('p.id');

		// Add the list ordering clause.
		$orderCol  = $this->state->get('list.ordering', 'p.ordering');
		$orderDirn = $this->state->get('list.direction', 'ASC');

		// Add the list ordering clause.
		$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

		return $query;
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @param   string $ordering  An optional ordering field.
	 * @param   string $direction An optional direction (asc|desc).
	 *
	 * @return  void
	 *
	 * @since   1.6
	 */
	protected function populateState($ordering = "p.id", $direction = "desc") {
		$app = JFactory::getApplication('administrator');

		$search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
		$this->setState('filter.search', $search);

		$statusId = $app->getUserStateFromRequest($this->context . '.filter.status_id', 'filter_status_id');
		$this->setState('filter.status_id', $statusId);

		$packageType = $app->getUserStateFromRequest($this->context.'.filter.package_type', 'filter_package_type', PACKAGE_TYPE_BUSINESS);
		$this->setState('filter.package_type', $packageType);

		// Check if the ordering field is in the white list, otherwise use the incoming value.
		$value = $app->getUserStateFromRequest($this->context . '.ordercol', 'filter_order', $ordering);
		$this->setState('list.ordering', $value);

		// Check if the ordering direction is valid, otherwise use the incoming value.
		$value = $app->getUserStateFromRequest($this->context . '.orderdir', 'filter_order_Dir', $direction);
		$this->setState('list.direction', $value);

		// List state information.
		parent::populateState($ordering, $direction);
	}
}
