<?php
/**
 * @package    JBusinessDirectory
 *
 * @author     CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2021 CMSJunkie. All rights reserved.
 * @license    https://www.gnu.org/licenses/agpl-3.0.en.html
 */

defined('_JEXEC') or die('Restricted access');
jimport('joomla.application.component.modeladmin');

use Joomla\Utilities\ArrayHelper;

/**
 * Package Model
 *
 */
class JBusinessDirectoryModelPackage extends JModelAdmin {
	/**
	 * @var        string    The prefix to use with controller messages.
	 * @since   1.6
	 */
	protected $text_prefix = 'COM_JBUSINESSDIRECTORY_PACKAGE';

	/**
	 * Model context string.
	 *
	 * @var        string
	 */
	protected $_context = 'com_jbusinessdirectory.package';

	/**
	 * Method to test whether a record can be deleted.
	 *
	 * @param   object    A record object.
	 *
	 * @return  boolean  True if allowed to delete the record. Defaults to the permission set in the component.
	 */
	protected function canDelete($record) {
		return true;
	}

	/**
	 * Method to test whether a record can be deleted.
	 *
	 * @param   object    A record object.
	 *
	 * @return  boolean  True if allowed to change the state of the record. Defaults to the permission set in the component.
	 */
	protected function canEditState($record) {
		return true;
	}

	/**
	 * Returns a Table object, always creating it
	 *
	 * @param   type    The table type to instantiate
	 * @param   string    A prefix for the table class name. Optional.
	 * @param   array  Configuration array for model. Optional.
	 *
	 * @return  JTable    A database object
	 */
	public function getTable($type = 'Package', $prefix = 'JTable', $config = array()) {
		return JTable::getInstance($type, $prefix, $config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @since   1.6
	 */
	protected function populateState() {
		$app = JFactory::getApplication('administrator');

		// Load the User state.
		$jinput = JFactory::getApplication()->input;
		if (!($packageType = $app->getUserState('com_jbusinessdirectory.packages.filter.package_type'))) {
			$packageType = $jinput->getInt('type', '1'); //package_type
		}
		$this->setState('package.type', $packageType);

		// Load the User state.
		$id = JFactory::getApplication()->input->getInt('id');
		$this->setState('package.id', $id);
	}

	/**
	 * Method to get a menu item.
	 *
	 * @param   integer    The id of the menu item to get.
	 *
	 * @return  mixed  Menu item data object on success, false on failure.
	 */
	public function &getItem($itemId = null) {
		$itemId = (!empty($itemId)) ? $itemId : (int) $this->getState('package.id');
		$false  = false;

		// Get a menu item row instance.
		$table = $this->getTable();

		// Attempt to load the row.
		$return = $table->load($itemId);

		// Check for a table object error.
		if ($return === false && $table->getError()) {
			$this->setError($table->getError());
			return $false;
		}

		$properties = $table->getProperties(1);
		$value      = ArrayHelper::toObject($properties, 'JObject');
		if (!empty($value->package_usergroup)) {
			$value->package_usergroup = explode(',', $value->package_usergroup);
		} else {
			$value->package_usergroup = array('1'); // public is by default selected as a package
		}

		if (empty($value->expiration_type)) {
			$value->expiration_type = 1;
		}

		if (!empty($value->price)) {
			$value->price = JBusinessUtil::convertPriceFromMysql($value->price);
		}

		if (!empty($value->renewal_price)) {
			$value->renewal_price = JBusinessUtil::convertPriceFromMysql($value->renewal_price);
		}

		if (!empty($value->special_price)) {
			$value->special_price = JBusinessUtil::convertPriceFromMysql($value->special_price);
		}

		if (!empty($value->trial_price)) {
			$value->trial_price = JBusinessUtil::convertPriceFromMysql($value->trial_price);
		}

		$value->special_from_date = JBusinessUtil::convertToFormat($value->special_from_date);
		$value->special_to_date   = JBusinessUtil::convertToFormat($value->special_to_date);

		return $value;
	}


	/**
	 * Method to get the menu item form.
	 *
	 * @param   array   $data     Data for the form.
	 * @param   boolean $loadData True if the form is to load its own data (default case), false if not.
	 *
	 * @return  JForm    A JForm object on success, false on failure
	 * @since   1.6
	 */
	public function getForm($data = array(), $loadData = true) {
		exit;
		// The folder and element vars are passed when saving the form.
		if (empty($data)) {
			$item = $this->getItem();
			// The type should already be set.
		}
		// Get the form.
		$form = $this->loadForm('com_jbusinessdirectory.package', 'item', array('control' => 'jform', 'load_data' => $loadData), true);
		if (empty($form)) {
			return false;
		}

		return $form;
	}

	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return  mixed  The data for the form.
	 * @since   1.6
	 */
	protected function loadFormData() {
		// Check the session for previously entered form data.
		$data = JFactory::getApplication()->getUserState('com_jbusinessdirectory.edit.package.data', array());

		if (empty($data)) {
			$data = $this->getItem();
		}

		return $data;
	}


	/**
	 * Method to save the form data.
	 *
	 * @param   array  The form data.
	 *
	 * @return  boolean  True on success.
	 */
	public function save($data) {
		$appSettings = JBusinessUtil::getApplicationSettings();		

		$id    = (!empty($data['id'])) ? $data['id'] : (int) $this->getState('package.id');
		$data["id"] = $id;
		$isNew = true;

		//trim space for the submitted values
		foreach ($data as &$item) {
			if (!is_array($item)) {
				$item = trim($item);
			}
		}

		if (!empty($data["package_usergroup"])) {
			$data["package_usergroup"] = implode(',', $data["package_usergroup"]);
		} else {
			$data["package_usergroup"] = '1';
		}
		
		if (!empty($data["expiration_type"]) && ($data["expiration_type"]==1 || $data["expiration_type"]==2)) {
			$data["trial_price"] = null;
			$data["trial_period_amount"] = null;
			$data["recurrence_count"] = null;
		}
		
		$data["special_from_date"] = JBusinessUtil::convertToMysqlFormat($data["special_from_date"]);
		$data["special_to_date"]   = JBusinessUtil::convertToMysqlFormat($data["special_to_date"]);
		if (!empty($data["price"])) {
			$data["price"] = JBusinessUtil::convertPriceToMysql($data["price"]);
		}
		if (!empty($data["renewal_price"])) {
			$data["renewal_price"] = JBusinessUtil::convertPriceToMysql($data["renewal_price"]);
		}
		if (!empty($data["special_price"])) {
			$data["special_price"] = JBusinessUtil::convertPriceToMysql($data["special_price"]);
		}
		if (!empty($data["trial_price"])) {
			$data["trial_price"] = JBusinessUtil::convertPriceToMysql($data["trial_price"]);
		}

		if (empty($data["trial_price"])) {
			$data["trial_price"] = 0;
		}
		if (empty($data["trial_period_amount"])) {
			$data["trial_period_amount"] = 0;
		}
		if (empty($data["recurrence_count"])) {
			$data["recurrence_count"] = 0;
		}

		if (empty($data["ordering"])) {
			$data["ordering"] = 0;
		}

		if (empty($data["special_price"])) {
			$data["special_price"] = 0;
		}
		if (empty($data["time_amount"])) {
			$data["time_amount"] = 0;
		}
		if (empty($data["renewal_price"])) {
			$data["renewal_price"] = 0;
		}

		$days = 0;
		switch ($data["time_unit"]) {
			case "D":
				$days = $data["time_amount"];
				break;
			case "W":
				$days = $data["time_amount"] * 7;
				break;
			case "M":
				$days = $data["time_amount"] * 30;
				break;
			case "Y":
				$days = $data["time_amount"] * 365;
				break;
			default:
				$days = $data["time_amount"];
		}

		$data["days"] = $days;

		$trialDays = 0;
		switch ($data["trial_period_unit"]) {
			case "D":
				$trialDays = $data["trial_period_amount"];
				break;
			case "W":
				$trialDays = $data["trial_period_amount"] * 7;
				break;
			case "M":
				$trialDays = $data["trial_period_amount"] * 30;
				break;
			case "Y":
				$trialDays = $data["trial_period_amount"] * 365;
				break;
			default:
				$trialDays = $data["trial_period_amount"];
		}

		$data["trial_days"] = $trialDays;

		switch ($data["expiration_type"]) {
			case "1":
				$data["days"]       = 0;
				$data["trial_days"] = 0;
				break;
			case "2":
				$data["trial_days"] = 0;
				break;
			case "3":
				$data["trial_days"] = 0;
				break;
		}

		$defaultLng  = JBusinessUtil::getLanguageTag();
		$jinput      = JFactory::getApplication()->input;

		$data['description'] = $jinput->get('description', '', 'RAW');

		$description = $jinput->get('description_' . $defaultLng, '', 'RAW');
		$priceDescription = isset($_REQUEST['price_description_' . $defaultLng])?$_REQUEST['price_description_' . $defaultLng]:""; 
		$name        = $jinput->get('name_' . $defaultLng, '', 'RAW');

		$languages = JBusinessUtil::getLanguages();
		if ($appSettings->enable_multilingual) {
			foreach ($languages as $lng) {
				$data['name_'. $lng] = isset($_REQUEST['name_'. ($lng)])?$_REQUEST['name_'. ($lng)]:"";
				$data['description_'. $lng] = isset($_REQUEST['description_'. ($lng)])?$_REQUEST['description_'. ($lng)]:"";
				$data['price_description_'. $lng] = isset($_REQUEST['price_description_'. ($lng)])?$_REQUEST['price_description_'. ($lng)]:"";
			}
		}

		$data['description'] = isset($_REQUEST['description'])?$_REQUEST['description']:"";
		if (empty($data["description"])) {
			$data["description"] = $description;
		}

		$data['price_description'] = isset($_REQUEST['price_description'])?$_REQUEST['price_description']:"";
		if (empty($data["price_description"])) {
			$data["price_description"] = $priceDescription;
		}

		if (!empty($name) && empty($data["name"])) {
			$data["name"] = $name;
		}

		

		// Get a row instance.
		$table = $this->getTable();

		// Load the row if saing an existing item.
		if ($id > 0) {
			$table->load($id);
			$isNew = false;
		}

		// Bind the data.
		if (!$table->bind($data)) {
			$this->setError($table->getError());
			return false;
		}

		// Check the data.
		if (!$table->check()) {
			$this->setError($table->getError());
			return false;
		}

		// Store the data.
		if (!$table->store(true)) {
			$this->setError($table->getError());
			return false;
		}

		$this->setState('package.id', $table->id);
		$table->insertRelations($table->id, $data["features"]);
		
		
		if ($appSettings->enable_multilingual) {
			JBusinessDirectoryTranslations::saveTranslations(PACKAGE_TRANSLATION, $table->id, 'description_', false, false, $data);
		}

		
		// Clean the cache
		$this->cleanCache();

		return true;
	}

	public function changeState($id) {
		$packagesTable = $this->getTable("Package");
		return $packagesTable->changeState($id);
	}

	public function changePopularState($id) {
		$this->populateState();
		$packagesTable = $this->getTable("Package");

		return $packagesTable->changePopularState($id);
	}

	public function getStates() {
		$states       = array();
		$state        = new stdClass();
		$state->value = 0;
		$state->text  = JTEXT::_("LNG_INACTIVE");
		$states[]     = $state;
		$state        = new stdClass();
		$state->value = 1;
		$state->text  = JTEXT::_("LNG_ACTIVE");
		$states[]     = $state;

		return $states;
	}

	public function getSelectedFeatures() {
		$packagesTable = $this->getTable("Package");
		$features      = $packagesTable->getSelectedFeatures($this->getState('package.id'));
		$result        = array();
		foreach ($features as $feature) {
			$result[] = $feature->feature;
		}

		return $result;
	}

	/**
	 * Method to delete groups.
	 *
	 * @param   array  An array of item ids.
	 *
	 * @return  boolean  Returns true on success, false on failure.
	 */
	public function delete(&$itemIds) {
		// Sanitize the ids.
		$itemIds = (array) $itemIds;
		ArrayHelper::toInteger($itemIds);
		$companiesTable = JTable::getInstance("Company", "JTable");
		$packageIds     = '';

		// Get a group row instance.
		$table = $this->getTable();

		// Iterate the items to delete each one.
		foreach ($itemIds as $itemId) {
			if ($companiesTable->getPackageUseStatus($itemId) == '0') {
				if (!$table->delete($itemId)) {
					$this->setError($table->getError());
					return false;
				}
				JBusinessDirectoryTranslations::deleteTranslationsForObject(PACKAGE_TRANSLATION, $itemId);
			} else {
				if ($packageIds == '') {
					$packageIds .= $itemId;
				} else {
					$packageIds .= ", " . $itemId;
				}
			}
		}

		if ($packageIds != '') {
			$this->setError(JText::_('LNG_PACKAGE_IN_USE') . " " . $packageIds);
			return false;
		}

		// Clean the cache
		$this->cleanCache();

		return true;
	}
}
