<?php
/**
 * @package    JBusinessDirectory
 *
 * @author CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2021 CMSJunkie. All rights reserved.
 * @license   https://www.gnu.org/licenses/agpl-3.0.en.html
 */

defined('_JEXEC') or die('Restricted access');
jimport('joomla.application.component.modellist');
require_once BD_CLASSES_PATH.'/attributes/attributeservice.php';
/**
 * List Model.
 *
 * @package     JBusinessDirectory
 * @subpackage  com_jbusinessdirectory
 */
class JBusinessDirectoryModelOffers extends JModelList {
	/**
	 * Constructor.
	 *
	 * @param   array  An optional associative array of configuration settings.
	 *
	 * @see     JController
	 * @since   1.6
	 */
	public function __construct($config = array()) {
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = array(
				'id', 'co.id',
				'name', 'co.subject',
				'companyName', 'bc.name',
				'publishStartDate', 'co.publish_start_date',
				'publishEndDate', 'co.publish_end_date',
				'startDate', 'co.startDate',
				'endDate', 'co.endDate',
				'viewCount', 'co.viewCount',
				'offerOfTheDay', 'co.offerOfTheDay',
				'featured', 'co.featured',
				'state', 'co.state',
				'approved', 'co.approved',
				'type', 'co.offer_type'
			);
		}

		parent::__construct($config);
	}

	/**
	 * Overrides the getItems method to attach additional metrics to the list.
	 *
	 * @return  mixed  An array of data items on success, false on failure.
	 *
	 * @since   1.6.1
	 */
	public function getItems() {
		$appSettings = JBusinessUtil::getApplicationSettings();
		$packagesTable = JTable::getInstance('Package', 'JTable', array());
		$offersTable = JTable::getInstance('Offer', 'JTable', array());

		// Get a storage key.
		$store = $this->getStoreId('getItems');

		// Try to load the data from internal storage.
		if (!empty($this->cache[$store])) {
			return $this->cache[$store];
		}

		// Load the list items.
		$items = parent::getItems();

		// If empty or an error, just return.
		if (empty($items)) {
			return array();
		}
		
		foreach ($items as $item) {
			$item->startDate = JBusinessUtil::convertToFormat($item->startDate);
			$item->endDate = JBusinessUtil::convertToFormat($item->endDate);

			$item->allow_offers = false;
			$item->expired = false;
			$item->not_visible = false;

			if (!$appSettings->enable_packages || $appSettings->item_decouple) {
				$item->allow_offers = true;
				$companyTable = JTable::getInstance('Company', 'JTable', array());
				if (!$appSettings->item_decouple) {
					$company = $companyTable->getPlainCompany($item->companyId);
					if (empty($company)) {
						$item->not_visible = false;
						$item->allow_offers = false;
					}
				}
			} elseif (!empty($item->companyId)) {
				$package = $packagesTable->getCurrentActivePackage($item->companyId);
				
				if (!empty($package->features)) {
					$item->features = $package->features;
				} else {
					$item->features = array();
				}

				if (in_array(COMPANY_OFFERS, $item->features)) {
					$item->allow_offers = true;
				}
			}

			if ((!JBusinessUtil::emptyDate($item->publish_end_date) && strtotime($item->publish_end_date) && (strtotime(date("Y-m-d")) > strtotime($item->publish_end_date)))
				|| (!JBusinessUtil::emptyDate($item->publish_start_date) && strtotime($item->publish_start_date) && (strtotime(date("Y-m-d")) < strtotime($item->publish_start_date)))) {
				$item->not_visible = true;
			}

			if ((!JBusinessUtil::emptyDate($item->endDate)) && strtotime($item->endDate) && (strtotime(date("Y-m-d")) > strtotime($item->endDate))) {
				$item->expired = true;
			}
		}
		
		// Add the items to the internal cache.
		$this->cache[$store] = $items;

		return $this->cache[$store];
	}

	/**
	 * Method to build an SQL query to load the list data.
	 *
	 * @return  string  An SQL query
	 *
	 * @since   1.6
	 */
	protected function getListQuery() {
		// Create a new query object.
		$db = $this->getDbo();
		$query = $db->getQuery(true);
		
		
		// Select all fields from the table.
		$query->select($this->getState('list.select', 'co.*'));
		$query->from($db->quoteName('#__jbusinessdirectory_company_offers').' AS co');
		
		// Join over the company types
		$query->select('bc.name as companyName');
		$query->join('LEFT', $db->quoteName('#__jbusinessdirectory_companies').' AS bc ON bc.id=co.companyId');

		// Join over the company types
		$query->select('et.name as offerType');
		$query->join('LEFT', $db->quoteName('#__jbusinessdirectory_company_offer_types').' AS et ON co.offer_type=et.id');

		// Join over the offer pictures
		$query->select('cop.picture_path, cop.picture_enable');
		$query->join('LEFT', $db->quoteName('#__jbusinessdirectory_company_offer_pictures').' AS cop ON cop.offerId=co.id
				and
						(cop.id in (
							select  min(op1.id) as min from #__jbusinessdirectory_company_offers co1
							left join  #__jbusinessdirectory_company_offer_pictures op1 on co1.id=op1.offerId
							where op1.picture_enable=1
							group by co1.id
						)
					)
				
				');
		
		// Filter by search in title.
		$search = $this->getState('filter.search');
		$search = trim($search);
		
		$keywords = array();
		if (strpos($search, '"') === false) {
			$keyword = $db->escape($search);
			$keywords = explode(" ", $keyword);
		} else {
			$keyword = trim($search, '"');
			$keyword = $db->escape($keyword);
			$keywords = array($keyword);
		}
		
		if (!empty($search)) {
			$query->where("(co.subject LIKE '%". implode("%' or co.subject LIKE '%", $keywords) ."%' or 
							bc.name LIKE '%". implode("%' or bc.name LIKE '%", $keywords) ."%')");
		}
		
		$listingId = $this->getState('filter.listing_id');
		if (is_numeric($listingId)) {
			$query->where('co.companyId ='.(int) $listingId);
		}
		
		$statusId = $this->getState('filter.status_id');
		if (is_numeric($statusId)) {
			$query->where('co.approved ='.(int) $statusId);
		}
	
		$stateId = $this->getState('filter.state_id');
		if (is_numeric($stateId)) {
			$query->where('co.state ='.(int) $stateId);
		}

		$itemType = $this->getState('filter.type');
		if (!empty($itemType)) {
			$query->where('co.item_type =' . (int) $itemType);
		}
	
		$query->group('co.id');

		// Add the list ordering clause.
		$orderCol  = $this->state->get('list.ordering', 'co.id');
		$orderDirn = $this->state->get('list.direction', 'DESC');
		
		$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

		return $query;
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 *
	 * @since   1.6
	 */
	protected function populateState($ordering = "co.id", $direction = 'desc') {
		$app = JFactory::getApplication('administrator');

		$listingId = $this->getUserStateFromRequest($this->context.'.filter.listing_id', 'listing_id');
		$this->setState('filter.listing_id', $listingId);
		
		$search = $this->getUserStateFromRequest($this->context.'.filter.search', 'filter_search');
		$this->setState('filter.search', $search);
		
		$statusId = $app->getUserStateFromRequest($this->context.'.filter.status_id', 'filter_status_id');
		$this->setState('filter.status_id', $statusId);
	
		$stateId = $app->getUserStateFromRequest($this->context.'.filter.state_id', 'filter_state_id');
		$this->setState('filter.state_id', $stateId);
	
		// Check if the ordering field is in the white list, otherwise use the incoming value.
		$value = $app->getUserStateFromRequest($this->context.'.ordercol', 'filter_order', $ordering);
		$this->setState('list.ordering', $value);
		
		// Check if the ordering direction is valid, otherwise use the incoming value.
		$value = $app->getUserStateFromRequest($this->context.'.orderdir', 'filter_order_Dir', $direction);
		$this->setState('list.direction', $value);

		$offerType = $app->getUserStateFromRequest($this->context.'.filter.type', 'filter_type', OFFER_TYPE_OFFER);
		$this->setState('filter.type', $offerType);
		
		// List state information.
		parent::populateState($ordering, $direction);
	}
	
	public function getCompanyTypes() {
		$companiesTable = $this->getTable("Company");
		return $companiesTable->getCompanyTypes();
	}
	
	public function getStates() {
		$states = array();
		$state = new stdClass();
		$state->value = 0;
		$state->text = JTEXT::_("LNG_INACTIVE");
		$states[] = $state;
		$state = new stdClass();
		$state->value = 1;
		$state->text = JTEXT::_("LNG_ACTIVE");
		$states[] = $state;
	
		return $states;
	}
	
	public function getStatuses() {
		$statuses = array();
		$status = new stdClass();
		$status->value = 0;
		$status->text = JTEXT::_("LNG_NEEDS_CREATION_APPROVAL");
		$statuses[] = $status;
		$status = new stdClass();
		$status->value = -1;
		$status->text = JTEXT::_("LNG_DISAPPROVED");
		$statuses[] = $status;
		$status = new stdClass();
		$status->value = 1;
		$status->text = JTEXT::_("LNG_APPROVED");
		$statuses[] = $status;
	
		return $statuses;
	}
	
	public function exportOffersCSVtoFile($path) {
		$csv_output = $this->getOffersCSV();
		$result =  file_put_contents($path, $csv_output);
		return $result;
	}
	
	
	public function getOffersCSV() {
		$jinput    = JFactory::getApplication()->input;
		$languages = JBusinessUtil::getLanguages();
		$delimiter = $jinput->getString("delimiter", ",");
		$includeId = $jinput->get("include_id");

		$offerAttributesTable = JTable::getInstance("OfferAttributes", "JTable");
		$offerTable = JTable::getInstance("Offer", "JTable");
		$attributesTable = JTable::getInstance("Attribute", "JTable");
		$attributes = $attributesTable->getAttributes(ATTRIBUTE_TYPE_OFFER);
		$categoryTable = JTable::getInstance("Category", "JBusinessTable");

		$csv_output = "";
		$name = "subject".$delimiter;
		$shortDesc = "short_description".$delimiter;
		$desc = "description".$delimiter;
		$metaName = "meta_title".$delimiter;
		$metaDesc = "meta_description".$delimiter;
		$metaKey = "meta_keywords".$delimiter;
		foreach ($languages as $lng) {
			$name.="subject_$lng".$delimiter;
			$shortDesc.="short_description_$lng".$delimiter;
			$desc.="description_$lng".$delimiter;
			$metaName.="meta_title_$lng".$delimiter;
			$metaDesc.="meta_description_$lng".$delimiter;
			$metaKey.="meta_keywords_$lng".$delimiter;
		}
		$metaKey = rtrim($metaKey, $delimiter);

		if (isset($includeId)) {
			$csv_output .= "id".$delimiter; 
		}

		$csv_output .= $name . "alias".$delimiter.$shortDesc.$desc."start_date".$delimiter
			."end_date".$delimiter."publish_start_date".$delimiter."publish_end_date".$delimiter."publish_start_time".$delimiter
			."publish_end_time".$delimiter."show_time".$delimiter."price".$delimiter."price_base".$delimiter."price_base_unit".$delimiter
			."special_price".$delimiter."special_price_base".$delimiter."special_price_base_unit".$delimiter."total_coupons".$delimiter
			."currency_id".$delimiter."company_id".$delimiter."state".$delimiter."categories".$delimiter."main_subcategory".$delimiter."street_number".$delimiter
			."address".$delimiter."area".$delimiter."country".$delimiter."city".$delimiter."province".$delimiter."county".$delimiter."postal_code".$delimiter."latitude".$delimiter
			."longitude".$delimiter."picture".$delimiter."offer_of_the_day".$delimiter."view_type".$delimiter
			."article_id".$delimiter."approved".$delimiter."view_count".$delimiter."item_type".$delimiter
			."featured".$delimiter."expiration_email_date".$delimiter."enable_offer_selling".$delimiter."min_purchase".$delimiter
			."max_purchase".$delimiter."quantity" .$delimiter.$metaName.$metaDesc.$metaKey;
		
		if (!empty($attributes)) {
			foreach ($attributes as $attribute) {
				$csv_output = $csv_output.$delimiter."attribute_".$attribute->name;
			}
		}
		$csv_output = $csv_output."\n";
		
		$start = 0;
		$batch = ITEMS_BATCH_SIZE;
		do {
			$offers = $offerTable->getOfferForExport($start, $batch);
			if (count($offers) > 0) {
				foreach ($offers as $offer) {
					$subCategory = "";
					if (!empty($offer->main_subcategory)) {
						$sub = $categoryTable->getCategoryById($offer->main_subcategory);
						if (isset($sub)) {
							$subCategory = $sub->name;
						}
					}

					$offer->short_description = str_replace(array("\r\n", "\r", "\n"), "<br />", $offer->short_description);
					$offer->short_description = str_replace('"', '""', $offer->short_description);
					$offer->description = str_replace(array("\r\n", "\r", "\n"), "<br />", $offer->description);
					$offer->description = str_replace('"', '""', $offer->description);
					$offer->meta_title = str_replace(array("\r\n", "\r", "\n"), "<br />", $offer->meta_title);
					$offer->meta_title = str_replace('"', '""', $offer->meta_title);
					$offer->meta_description = str_replace(array("\r\n", "\r", "\n"), "<br />", $offer->meta_description);
					$offer->meta_description = str_replace('"', '""', $offer->meta_description);

					$translations = JBusinessDirectoryTranslations::getAllTranslations(OFFER_DESCRIPTION_TRANSLATION, $offer->id);
					$translationsMeta = JBusinessDirectoryTranslations::getAllTranslations(OFFER_META_TRANSLATION, $offer->id);

					$name = "\"$offer->subject\"" . $delimiter;
					$shortDesc = "\"$offer->short_description\"" . $delimiter;
					$desc = "\"$offer->description\"" . $delimiter;
					$metaName = "\"$offer->meta_title\"" . $delimiter;
					$metaDesc = "\"$offer->meta_description\"" . $delimiter;
					$metaKey = "\"$offer->meta_keywords\"" . $delimiter;
					foreach ($languages as $lng) {
						$langContentName = isset($translations[$lng . "_name"]) ? $translations[$lng . "_name"] : "";
						$langContentShort = isset($translations[$lng . "_short"]) ? $translations[$lng . "_short"] : "";
						$langContentDesc = isset($translations[$lng]) ? $translations[$lng] : "";
						$contentNameMeta = isset($translationsMeta[$lng . "_name"]) ? $translationsMeta[$lng . "_name"] : "";
						$contentDescMeta = isset($translationsMeta[$lng]) ? $translationsMeta[$lng] : "";
						$contentKeyMeta = isset($translationsMeta[$lng . "_short"]) ? $translationsMeta[$lng . "_short"] : "";

						$name .= "\"".$langContentName."\"".$delimiter;
						$shortDesc .= "\"".$langContentShort."\"".$delimiter;
						$desc .= "\"".$langContentDesc."\"".$delimiter;
						$metaName .= "\"".$contentNameMeta."\"".$delimiter;
						$metaDesc .= "\"".$contentDescMeta."\"".$delimiter;
						$metaKey .= "\"".$contentKeyMeta."\"".$delimiter;
					}
					$metaKey = rtrim($metaKey, $delimiter);

					
					if (isset($includeId)) {
						$csv_output .= "\"$offer->id\"".$delimiter; 
					}

					$csv_output .= $name . "\"" . $offer->alias . "\"" . $delimiter . $shortDesc
						. $desc . "\"$offer->startDate\"" . $delimiter . "\"$offer->endDate\"" . $delimiter
						. "\"$offer->publish_start_date\"" . $delimiter . "\"$offer->publish_end_date\"" . $delimiter . "$offer->publish_start_time" . $delimiter
						. "\"$offer->publish_end_time\"" . $delimiter . "\"$offer->show_time\"" . $delimiter . "\"$offer->price\"" . $delimiter
						. "\"$offer->price_base\"" . $delimiter . "\"$offer->price_base_unit\"" . $delimiter . "\"$offer->specialPrice\"" . $delimiter
						. "\"$offer->special_price_base\"" . $delimiter . "\"$offer->special_price_base_unit\"" . $delimiter
						. "\"$offer->total_coupons\"" . $delimiter . "\"$offer->currencyId\"" . $delimiter . "\"$offer->companyId\"" . $delimiter
						. "\"$offer->state\"" . $delimiter . "\"$offer->categories\"" . $delimiter . "\"$subCategory\"" .$delimiter ."\"$offer->street_number\"" . $delimiter
						. "\"$offer->address\"" . $delimiter . "\"$offer->area\"" .$delimiter."\"$offer->countryName\"" . $delimiter . "\"$offer->city\"" . $delimiter . "\"$offer->province\"" . $delimiter
						. "\"$offer->county\""  .$delimiter."\"$offer->postalCode\""  . $delimiter . "\"$offer->latitude\"" . $delimiter . "\"$offer->longitude\"" . $delimiter . "\"$offer->pictures\"" . $delimiter
						. "\"$offer->offerOfTheDay\"" . $delimiter . "\"$offer->view_type\"" . $delimiter . "\"$offer->article_id\"" . $delimiter
						. "\"$offer->approved\"" . $delimiter . "\"$offer->viewCount\"" . $delimiter. "\"$offer->item_type\"" . $delimiter . "\"$offer->featured\"" . $delimiter
						. "\"$offer->expiration_email_date\"" . $delimiter . "\"$offer->enable_offer_selling\"" . $delimiter . "\"$offer->min_purchase\"" . $delimiter
						. "\"$offer->max_purchase\"" . $delimiter . "\"$offer->quantity\"" . $delimiter . $metaName
						. $metaDesc . $metaKey;

					$offerAttributes = $offerAttributesTable->getOfferAttributes($offer->id);
					foreach ($attributes as $attribute) {
						$found = false;
						foreach ($offerAttributes as $key => $companyAttribute) {
							if ($attribute->code == $companyAttribute->code) {
								$attributeValue = AttributeService::getAttributeValues($companyAttribute);
								$csv_output .= $delimiter."\"$attributeValue\"";
								$found = true;
								unset($offerAttributes[$key]);
								break;
							}
						}
						if (!$found) {
							$csv_output .= $delimiter."\"\"";
						}
					}
					$csv_output .= "\n";
				}
			}
			$start += $batch;
		} while (count($offers) == $batch);
	
		return $csv_output;
	}

	public function exportOffersCSV() {
		$csv_output = $this->getOffersCSV();

		$fileName = "jbusinessdirectory_offers";
		header("Content-type: application/vnd.ms-excel");
		header("Content-disposition: csv" . date("Y-m-d") . ".csv");
		header("Content-disposition: filename=".$fileName.".csv");
		print $csv_output;
	}
	
	public function getOffersWithTranslationCSV() {
		$jinput    = JFactory::getApplication()->input;
		$delimiter = $jinput->getString("delimiter", ",");
	
		$offerTable = $this->getTable("Offer", 'JTable');
	
		$languages = JBusinessUtil::getLanguages();
	
		$csv_output = "ID".$delimiter."Name".$delimiter;
	
	
		$csv_output .= "Company_id".$delimiter."Currency_id".$delimiter."Short_Description".$delimiter;
	
		foreach ($languages as $language) {
			$csv_output .= "Short_Description $language".$delimiter;
		}
	
		$csv_output .= "Description".$delimiter;
	
		foreach ($languages as $language) {
			$csv_output .= "Description $language".$delimiter;
		}
	
		$csv_output .= "CategoriesIds".$delimiter."Categories".$delimiter."Picture".$delimiter."Price".$delimiter."Special_Price".$delimiter."Price_Base".$delimiter."Price_Base_Unit".$delimiter."Special_Price_Base".$delimiter."Special_Price_Base_Unit".$delimiter."Total_Coupons".$delimiter."Start_Date".$delimiter."End_Date".$delimiter."State".$delimiter."Approved".$delimiter."Offer_Of_The_Day".$delimiter."View_Count".$delimiter."Full_Address".$delimiter."City".$delimiter."Country".$delimiter."Publish_Start_Date".$delimiter."Publish_End_Date".$delimiter."View_Type".$delimiter."URL".$delimiter."Article_Id".$delimiter."Latitude".$delimiter."Longitude".$delimiter."Featured".$delimiter."Created".$delimiter."Show_Time".$delimiter."Publish_Start_Time".$delimiter."Publish_End_Time".$delimiter."Expiration_Email_Date".$delimiter."Main_Category_ID".$delimiter."Enable_offer_selling".$delimiter."Min_Purchase".$delimiter."Max_Purchase".$delimiter."Quantity";
	
		$csv_output = $csv_output."\n";

		$start = 0;
		$batch = ITEMS_BATCH_SIZE;
		do {
			$offers = $offerTable->getOfferForExport($start, $batch);
			if (count($offers)>0) {
				foreach ($offers as $offer) {
					$offer->short_description = str_replace(array("\r\n", "\r", "\n"), "<br />", $offer->short_description);
					$offer->description = str_replace(array("\r\n", "\r", "\n"), "<br />", $offer->description);
					$offer->description = str_replace('"', '""', $offer->description);
					$csv_output .= $offer->id . $delimiter . "\"$offer->subject\"" . $delimiter;

					$csv_output .= "\"$offer->companyId\"" . $delimiter . "\"$offer->currencyId\"" . $delimiter . "\"$offer->short_description\"" . $delimiter;

					foreach ($languages as $lng) {
						$content_short = array();
						$translation = JBusinessDirectoryTranslations::getObjectTranslation(OFFER_DESCRIPTION_TRANSLATION, $offer->id, $lng);
						$content_short[] = empty($translation) ? "" : $translation->name;
						if (empty($content_short)) {
							$csv_output .= "" . $delimiter;
						} else {
							$csv_output .= "\"" . implode(",", $content_short) . "\"" . $delimiter;
						}
					}

					$csv_output .= "\"$offer->description\"" . $delimiter;
					foreach ($languages as $lng) {
						$content = array();
						$translation = JBusinessDirectoryTranslations::getObjectTranslation(OFFER_DESCRIPTION_TRANSLATION, $offer->id, $lng);
						$content[] = empty($translation) ? "" : $translation->name;
						if (empty($content)) {
							$csv_output .= "" . $delimiter;
						} else {
							$csv_output .= "\"" . implode(",", $content) . "\"" . $delimiter;
						}
					}

					$csv_output .= "\"$offer->categoriesIds\"" . $delimiter;
					$csv_output .= "\"$offer->categories\"" . $delimiter;

					$pictures = explode(",", $offer->pictures);
					foreach ($pictures as &$picture) {
						$picture = BD_PICTURES_PATH . $picture;
					}
					$pictures = implode(",", $pictures);
					$csv_output .= "\"" . $pictures . "\"";

					$csv_output .= $delimiter . "\"$offer->price\"" . $delimiter . "\"$offer->specialPrice\"" . $delimiter . "\"$offer->price_base\"" . $delimiter . "\"$offer->price_base_unit\"" . $delimiter . "\"$offer->special_price_base\"" . $delimiter . "\"$offer->special_price_base_unit\"" . $delimiter . "$offer->total_coupons" . $delimiter . "\"$offer->startDate\"" . $delimiter . "\"$offer->endDate\"" . $delimiter . "\"$offer->state\"" . $delimiter . "\"$offer->approved\"" . $delimiter . "\"$offer->offerOfTheDay\"" . $delimiter . "\"$offer->viewCount\"" . $delimiter . "\"$offer->address\"" . $delimiter . "\"$offer->city\"" . $delimiter . "\"$offer->county\"" . $delimiter . "\"$offer->publish_start_date\"" . $delimiter . "\"$offer->publish_end_date\"" . $delimiter . "\"$offer->view_type\"" . $delimiter . "\"$offer->url\"" . $delimiter . "\"$offer->article_id\"" . $delimiter . "\"$offer->latitude\"" . $delimiter . "\"$offer->longitude\"" . $delimiter . "\"$offer->featured\"" . $delimiter . "\"$offer->created\"" . $delimiter . "\"$offer->show_time\"" . $delimiter . "\"$offer->publish_start_time\"" . $delimiter . "\"$offer->publish_end_time\"" . $delimiter . "\"$offer->expiration_email_date\"" . $delimiter . "\"$offer->main_subcategory\"" . $delimiter . "\"$offer->enable_offer_selling\"" . $delimiter . "\"$offer->min_purchase\"" . $delimiter . "\"$offer->max_purchase\"" . $delimiter . "\"$offer->quantity\"";

					$csv_output .= "\n";
				}
			}
			$start += $batch;
		} while (count($offers) == $batch);
	
		return $csv_output;
	}
}
