<?php
/**
 * @package    JBusinessDirectory
 *
 * @author CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2021 CMSJunkie. All rights reserved.
 * @license   https://www.gnu.org/licenses/agpl-3.0.en.html
 */
defined('_JEXEC') or die('Restricted access');

use Joomla\Utilities\ArrayHelper;

JTable::addIncludePath(DS.'components'.'com_jbusinessdirectory'.DS.'tables');

JModelLegacy::addIncludePath(JPATH_SITE . '/components/com_jbusinessdirectory/models', 'Orders');

class JBusinessDirectoryModelPayment extends JModelLegacy {
	public function __construct() {
		parent::__construct();
		$this->appSettings = JBusinessUtil::getApplicationSettings();
	}
	
	/**
	 * Populate state
	 * @param unknown_type $ordering
	 * @param unknown_type $direction
	 */
	protected function populateState($ordering = null, $direction = null) {
		$app = JFactory::getApplication('administrator');
		$id = JFactory::getApplication()->input->getInt('orderId');
		$this->setState('payment.orderId', $id);
		
		$paymentMethod = JFactory::getApplication()->input->get('payment_method');
		$this->setState('payment.payment_method', $paymentMethod);
	}
	
	/**
	 * Returns a Table object, always creating it
	 *
	 * @param   type	The table type to instantiate
	 * @param   string	A prefix for the table class name. Optional.
	 * @param   array  Configuration array for model. Optional.
	 * @return  JTable	A database object
	 */
	public function getTable($type = 'PaymentProcessors', $prefix = 'JTable', $config = array()) {
		return JTable::getInstance($type, $prefix, $config);
	}
	
	/**
	 * Get current order
	 * @return unknown
	 */
	public function getOrder() {
		$model = JModelLegacy::getInstance('Orders', 'JBusinessDirectoryModel', array('ignore_request' => true));
		$order = $model->getOrder($this->getState('payment.orderId'));
		
		if (!empty($order)) {
			$companiesTable = $this->getTable("Company");
			$order->company = $companiesTable->getCompany($order->company_id);
		}
		
		return $order;
	}
	
	public function applyDiscount() {
		$model = JModelLegacy::getInstance('Orders', 'JBusinessDirectoryModel', array('ignore_request' => true));
		$discountCode = JFactory::getApplication()->input->get("discount_code");
		$result = $model->applyDiscount($this->getState('payment.orderId'), $discountCode);
		
		return $result;
	}

	/**
	 * Send payment e-mail
	 * @param unknown_type $data
	 */
	public function sendPaymentEmail($data) {
		$orderTable = JTable::getInstance("Order", "JTable", array());
		$orderTable->load($data->order_id);
		
		$properties = $orderTable->getProperties(1);
		$order = ArrayHelper::toObject($properties, 'JObject');
		$order->details = $data;
		
		if ($order->amount==0) {
			$order->details->processor_type = JText::_("LNG_NO_PAYMENT_INFO_REQUIRED");
		}
		
		$companiesTable = $this->getTable("Company");
		$company = $companiesTable->getCompany($order->company_id);

		if (!empty($company->userId)) {
			$billingDetailsTable = JTable::getInstance("BillingDetails", "JTable", array());
			$order->billingDetails = $billingDetailsTable->getBillingDetails($company->userId);
		}
		
		$packageTable =  $this->getTable("Package");
		$order->package = $packageTable->getPackage($order->package_id);

		$taxesTable = JTable::getInstance("Taxes", "Table", array());
		$order->taxes = $taxesTable->getTaxes();
		if (!empty($order->taxes)) {
			$appSettings = JBusinessUtil::getApplicationSettings();
			foreach ($order->taxes as &$tax) {
				if ($appSettings->enable_multilingual) {
					JBusinessDirectoryTranslations::updateEntityTranslation($tax, TAX_DESCRIPTION_TRANSLATION);
				}
				if ($tax->tax_type == 2) {
					$tax->percentage = $tax->tax_amount;
					$tax->tax_amount = $tax->tax_amount * $order->initial_amount / 100;
				}
			}
		}
		$order->orderDetails = $this->prepareOrderEmail($order);
	
		return EmailService::sendPaymentEmail($company, $order);
	}

	/**
	 * Prepare a html with detail for order
	 *
	 * @param $taxes object contain order details
	 * @return string string return a html string for the order
	 */
	public static function prepareOrderEmail($order) {
		$appSettings = JBusinessUtil::getApplicationSettings();

		$result = "<div>";
		$result .= "<div class=\"payment-items\">";
		$result  .= "<table border=\"0px\" cellpadding=\"5\">";
		$result .= "<thead>";
		$result .= "<tr bgcolor=\"#ececec\">";
		$result .= "<td>".JText::_('LNG_PRODUCT_SERVICE')."</td>";
		$result .= "<td>".JText::_('LNG_UNIT_PRICE')."</td>";
		$result .= "<td>".JText::_('LNG_TOTAL')."</td>";
		$result .= "</tr>";
		$result .= "</thead>";

		$result .= "<tbody>";
		$result .= "<tr>";
		$result .= "<td align=\"left\">";
		$result .= "<div class=\"left\">";
		$result .= "<strong>".$order->service ."</strong><br/>";
		$result .= $order->description;
		$result .= "</div>";
		$result .= "</td>";
		$result .= "<td align=\"left\">".JBusinessUtil::getPriceFormat($order->initial_amount)."</td>";
		$result .= "<td align=\"left\">".JBusinessUtil::getPriceFormat($order->initial_amount)."</td>";
		$result .= "</tr>";
		if ($order->discount_amount>0) {
			$result .= "<tr>";
			$result .= "<td></td>";
			$result .= "<td align=\"left\"><b>".JText::_("LNG_DISCOUNT")."</b></td>";
			$result .= "<td align=\"left\">".JBusinessUtil::getPriceFormat($order->discount_amount)."</td>";
			$result .= "</tr>";
		}
		$result .= "<tr>";
		$result .= "<td></td>";
		$result .= "<td align=\"left\"><b>".JText::_("LNG_SUB_TOTAL")."</b></td>";
		$result .= "<td align=\"left\"><b>".JBusinessUtil::getPriceFormat($order->initial_amount - $order->discount_amount)."</td>";
		$result .= "</tr>";
		if ($appSettings->vat>0) {
			$result .= "<tr>";
			$result .= "<td></td>";
			$result .= "<td align=\"left\"><b>".JText::_("LNG_VAT");
			$result .= "(".$appSettings->vat."%)";
			$result .= "</b></td>";
			$result .= "<td align=\"left\">".JBusinessUtil::getPriceFormat($order->vat_amount)."</td>";
			$result .= "</tr>";
		}
		if (!empty($order->taxes)) {
			foreach ($order->taxes as $tax) {
				$result .= "<tr>";
				$result .= "<td></td>";
				$result .= "<td>";
				$result .= $tax->tax_name;
				if ($tax->tax_type==2) {
					$result .= "(".$tax->percentage."%)";
				} else {
					$result .= " ";
				}
				$result .= "</td>";
				$result .= "<td align=\"left\">".JBusinessUtil::getPriceFormat($tax->tax_amount)."</td>";
				$result .= "</tr>";
			}
		}
		$result .= "<tr>";
		$result .= "<td>&nbsp;</td>";
		$result .= "<td align=\"left\"><b>".JText::_('LNG_TOTAL')."</b></td>";
		$result .= "<td align=\"left\">".JBusinessUtil::getPriceFormat($order->amount)."</td>";
		$result .= "</tr>";
		$result .= "</tbody>";
		$result .= "</table>";
		$result .= "</div></div>";


		return $result;
	}

	/**
	 * Send payment details email
	 *
	 * @param unknown $data
	 * @return void|boolean|number|unknown
	 */
	public function sendPaymentDetailsEmail($data) {
		$orderTable = JTable::getInstance("Order", "JTable", array());
		$orderTable->load($data->order_id);
	
		$properties = $orderTable->getProperties(1);
		$order = ArrayHelper::toObject($properties, 'JObject');
		$order->details = $data;
		$order->orderDetails = $this->prepareOrderEmail($order);
		
		$companiesTable = $this->getTable("Company");
		$company = $companiesTable->getCompany($order->company_id);
		
		if (!empty($company->userId)) {
			$billingDetailsTable = JTable::getInstance("BillingDetails", "JTable", array());
			$order->billingDetails = $billingDetailsTable->getBillingDetails($company->userId);
		}
		
		$packageTable =  $this->getTable("Package");
		$order->package = $packageTable->getPackage($order->package_id);

		$taxesTable = JTable::getInstance("Taxes", "Table", array());
		$order->taxes = $taxesTable->getTaxes();
		if ($this->appSettings->enable_multilingual) {
			JBusinessDirectoryTranslations::updateEntityTranslation($order->taxes, TAX_DESCRIPTION_TRANSLATION);
		}
		if (!empty($order->taxes)) {
			foreach ($order->taxes as &$tax) {
				if ($tax->tax_type == 2) {
					$tax->percentage = $tax->tax_amount;
					$tax->tax_amount = $tax->tax_amount * $order->initial_amount / 100;
				}
			}
			$order->taxesDetails = $this->prepareOrderEmail($order);
		}
	
		return EmailService::sendPaymentDetailsEmail($company, $order);
	}
}
