<?php
/**
 * @package    JBusinessDirectory
 *
 * @author CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2021 CMSJunkie. All rights reserved.
 * @license   https://www.gnu.org/licenses/agpl-3.0.en.html
 */
defined('_JEXEC') or die('Restricted access');
jimport('joomla.application.component.modelitem');
JTable::addIncludePath(DS.'components'.DS.JFactory::getApplication()->input->get('option').DS.'tables');
require_once(HELPERS_PATH.'/category_lib.php');
require_once(JPATH_COMPONENT_ADMINISTRATOR.DS.'models'.DS.'offercoupon.php');
require_once(BD_CLASSES_PATH.'/attributes/attributeservice.php');

class JBusinessDirectoryModelOffer extends JModelItem {
	public $offer = null;
	
	public function __construct() {
		parent::__construct();
		
		$this->context="com_jbusinessdirectory.offer.details";
		
		$this->appSettings = JBusinessUtil::getApplicationSettings();
		$this->offerId = JFactory::getApplication()->input->get('offerId');
		$this->offerId = intval($this->offerId);
	}

	/**
	 * Returns a Table object, always creating it
	 *
	 * @param   type	The table type to instantiate
	 * @param   string	A prefix for the table class name. Optional.
	 * @param   array  Configuration array for model. Optional.
	 * @return  JTable	A database object
	 */
	public function getTable($type = 'Offer', $prefix = 'JTable', $config = array()) {
		return JTable::getInstance($type, $prefix, $config);
	}
	
	
	/**
	 * Method to get a cache id based on the listing id.
	 *
	 * @param unknown $params
	 * @param string $id
	 * @return string
	 */
	protected function getCacheId($id) {
		return md5($this->context . ':' . $id);
	}

	
	/**
	 * Get offer based on offer id from cache or from database
	 *
	 * @return offer data
	 */
	
	public function getOffer() {
		if (empty($this->offerId)) {
			return;
		}
		
		$offerData = null;
		$cacheIdentifier = $this->getCacheId($this->offerId);
		try {
			if ($this->appSettings->enable_cache) {
				$cache = JCache::getInstance();
				$offerData = $cache->get($cacheIdentifier);
				if (empty($offerData)) {
					$offerData = $this->getOfferData();
					$cache->store($offerData, $cacheIdentifier);
				}
			} else {
				$offerData = $this->getOfferData();
			}
		} catch (RuntimeException $e) {
			$this->setError($e->getMessage());
			return null;
		}


		
		return $offerData;
	}


	/**
	 * Get item function
	 *
	 * @return void
	 */
	public function getItem($pk = NULL){
		return $this->getOffer();
	}

	/**
	 * Retrieve the merchants related to this particular product
	 *
	 * @return mixed
	 */
	public function getMerchants($offerId) {
		$table = $this->getTable("ProductMerchants");

		$searchDetails = array();
		$searchDetails["productId"] = $offerId;

		$companies = $table->getMerchantsDetails($searchDetails);
		foreach ($companies as $company) {
			$pictures = explode(",", $company->pictures);
			array_walk($pictures, function (&$value, $key) use ($pictures) {
				$item = new stdClass();
				$item->picture_path = $pictures[$key];
				$value = $item;
			});
			$company->pictures = $pictures;
		}
		return $companies;
	}
	
	

	public function getOfferData() {
		$offersTable = JTable::getInstance("Offer", "JTable");
		$offer = $offersTable->getActiveOffer($this->offerId);
		if (empty($offer)) {
			return $offer;
		}

		$this->offer = $offer;
		$offer->specialPrice = (float)$offer->specialPrice;
		$offer->price = (float)$offer->price;
		
		$offer->pictures = $offersTable->getOfferPictures($this->offerId);
		$this->increaseViewCount($this->offerId);
		if (!empty($offer->companyId)) {
			$companiesTable = JTable::getInstance("Company", "JTable");
			$company = $companiesTable->getCompany($offer->companyId);
			
			if (!empty($company->pictures)) {
				$pictures = explode(",", $company->pictures);
				array_walk($pictures, function (&$value, $key) use ($pictures) {
					$item = new stdClass();
					$item->picture_path = $pictures[$key];
					$value = $item;
				});
				$company->pictures = $pictures;
			}
			
			$offer->company=$company;
			
			$offer->companyPackage = $this->getPackage($offer->companyId);
		}
		
		$offer->checkOffer = $this->checkOffer();
		
		if ($this->appSettings->enable_multilingual) {
			JBusinessDirectoryTranslations::updateEntityTranslation($offer, OFFER_DESCRIPTION_TRANSLATION);
			
			if (!empty($offer->company)) {
				JBusinessDirectoryTranslations::updateEntityTranslation($offer->company, BUSSINESS_DESCRIPTION_TRANSLATION);
			}
			JBusinessDirectoryTranslations::getInstance()->updateMetaDataTranslation($offer, OFFER_META_TRANSLATION);
		}

		$offer->attachments = JBusinessDirectoryAttachments::getAttachments(OFFER_ATTACHMENTS, $this->offerId, true);
		if (!empty($offer->attachments)) {
			$offer->attachments = array_slice($offer->attachments, 0, $this->appSettings->max_attachments);
			foreach ($offer->attachments as $attach) {
				$attach->properties = JBusinessUtil::getAttachProperties($attach);
			}
		}
		$attributeConfig = JBusinessUtil::getAttributeConfiguration(DEFAULT_ATTRIBUTE_TYPE_LISTING);
		$offer->company = JBusinessUtil::updateItemDefaultAtrributes($offer->company, $attributeConfig);
		$attributeOfferConfig = JBusinessUtil::getAttributeConfiguration(DEFAULT_ATTRIBUTE_TYPE_OFFER);
		$offer = JBusinessUtil::updateItemDefaultAtrributes($offer, $attributeOfferConfig);

		$userId = JBusinessUtil::getUser()->id;
		$offer->isBookmarked = false;
		if (!empty($userId)) {
			$bookmarkTable = $this->getTable('Bookmark');
			$offer->bookmark = $bookmarkTable->getBookmark($offer->id, $userId, BOOKMARK_TYPE_OFFER);
		}

		if (!empty($offer->categories)) {
			$offer->categories = explode('#|', $offer->categories);
			foreach ($offer->categories as $k=>&$category) {
				$category = explode("|", $category);
			}
		}

		$maxCategories = !empty($offer->categories)?count($offer->categories):0;
		if (!empty($this->appSettings->max_categories)) {
			$maxCategories = $this->appSettings->max_categories;
		}

		if (!empty($offer->categories)) {
			$offer->categories = array_slice($offer->categories, 0, $maxCategories);
		}

		if ($offer->item_type == OFFER_TYPE_PRODUCT) {
			$offer->merchants = $this->getMerchants($offer->id);
		}
		$offer->sellingOptions = $this->getSellingOptionsFrontEnd($offer->id,$offer->main_subcategory);
		$stockTable = $this->getTable('OfferStock');
		$stockQuantity = $stockTable->getOfferQuantity($offer->id, $offer->main_subcategory)->quantity;
		$offer->quantity = !empty($stockQuantity)?$stockQuantity:$offer->quantity;

		$offer->paymentMethods = PaymentService::getPaymentProcessors(true, JBD_APP_SELL_OFFERS, $offer->companyId);

		$_REQUEST["offer-data"] = $offer;
		
		//dispatch load offer
		JFactory::getApplication()->triggerEvent('onAfterJBDLoadOffer', array($offer));
		
		return $offer;
	}

	/**
	 * Get the selling options for the item based on its main category and creates the structure that will be used to
	 * render the selling options on item detail view on front end
	 * The structure has this format
	 *
	 * @param $itemId int item ID
	 *
	 * @return array
	 *
	 * @since version
	 */
	public function getSellingOptionsFrontEnd($itemId,$mainCategory){

		$categoryService = new JBusinessDirectorCategoryLib();
		$fullCategory = $categoryService->getCompleteCategoryById($mainCategory,CATEGORY_TYPE_OFFER);
		$stockTable = $this->getTable('OfferStock');

		//check if category is parent or subcategory. If its sub than add also category of level 1 to take the custom attributes
		if (!empty($fullCategory) && isset($fullCategory['level'])) {
			if ($fullCategory['level'] == 1) {
				$stockCategory = $mainCategory;
			} else {
				$stockCategory = reset($fullCategory['path'])[0] . ',' . $mainCategory;
			}
		}else{
			$stockCategory = null;
		}
		$stocks = $stockTable->getOfferStocks($itemId, $mainCategory, false);
		$stocksConfig = array();
		$attributesWithQuantities = array();
		if (!empty($stocks)) {
			$strockIds = array();
			//filter the stocks and determine which attribute has stock option defined
			foreach ($stocks as $stock){
				$strockIds[] = $stock->id;
				$attrCombinations = explode('##',$stock->attributes);
				foreach ($attrCombinations as $combination){
					if (!isset($attributesWithQuantities[explode('_',$combination)[0]])){
						$attributesWithQuantities[explode('_',$combination)[0]] = array(explode('_',$combination)[1]);
					}else{
						if (!in_array(explode('_',$combination)[1], $attributesWithQuantities[explode('_',$combination)[0]])) {
							$attributesWithQuantities[explode('_',$combination)[0]][] = explode('_',$combination)[1];
						}
					}
				}
			}

			//get offer stock configs
			$stocksConfig = $stockTable->getOfferStocksConfig($itemId, $stocks[0]->id, $stockCategory);
		}
		if (!empty($stocksConfig)){
			//filter all stock configs and remove the ones tha do not have stock options defined
			foreach ($stocksConfig as &$item){
				$attributeOptions = explode("|#", $item->options);
				$attributeOptionsIDS = explode("|#", $item->optionsIDS);
				foreach ($attributeOptionsIDS as $key => $attributeOptionsID){
					if (!in_array($attributeOptionsID,$attributesWithQuantities[$item->id])){
						unset($attributeOptionsIDS[$key]);
						unset($attributeOptions[$key]);
					}
				}
				$item->options = implode('|#',$attributeOptions);
				$item->optionsIDS = implode('|#',$attributeOptionsIDS);
			}
		}

		return $stocksConfig;
	}

	/**
	 * Method that retrieves the quantity for selling options and create the quantity select box html
	 *
	 * @param $data array contains the data that needs to be queried
	 *
	 * @return mixed
	 *
	 * @since version
	 */
	public function updateQuantityAjax($data)
	{
		if (!isset($data['mainCatId'])){
			$data['mainCatId'] = 0;
		}

		$table = JTable::getInstance("OfferStockConfig");
		//check if search has been reset. If yes then reset all select filters after the first one
		if($data['oldVal'] != $data['newValue']){
			foreach ($data['selectedValues'] as &$val){
				if ($val != $data['newValue']){
					$val = "";
				}
			}
		}
		//if we have search values selected than do a search to get the stock configurations
		$newSelectedValues = array_values(array_filter($data['selectedValues']));
		if (!empty($newSelectedValues)) {
			$configurations = $table->getRelatedOfferStock($data['offerId'],$data['selectedValues']);
		}else{
			$configurations = array();
		}

		// get all item selling options
		$sellingOptions = $this->getSellingOptionsFrontEnd($data['offerId'],$data['mainCatId']);
		$allowedOptions = array();
		//create the allowed options array for what the user has searched
		foreach ($data['selectedValues'] as $val){
			if (!empty($val)) {
				$allowedOptions[explode('_',$val)[0]][] = explode('_',$val)[1];
			}
		}
		foreach ($configurations as $configuration){
			$attrOpts = explode(',',$configuration->attrCombinations);
			foreach ($attrOpts as $attrOpt){
				$idValAttr = explode('_',$attrOpt);
				if (!isset($allowedOptions[$idValAttr[0]])){
					$allowedOptions[$idValAttr[0]][] = $idValAttr[1];
				}else{
					if (!in_array($idValAttr[1],$allowedOptions[$idValAttr[0]])) {
						$allowedOptions[$idValAttr[0]][] = $idValAttr[1];
					}
				}
			}
		}

		//foreach selling options we remove the options that do not have a quantity or that are not part of user search
		//except the first attribute option which will have always all the options that have combinations, because it
		//will be used to reset also the search
		foreach ($sellingOptions as &$sellingOption){
			if (!empty($allowedOptions)){
				if ($sellingOption->id != explode('_',reset($data['selectedValues']))[0]) {
					$attributeOptions = explode("|#", $sellingOption->options);
					$attributeOptionsIDS = explode("|#", $sellingOption->optionsIDS);
					foreach ($attributeOptionsIDS as $key => $option) {
						if (!isset($allowedOptions[$sellingOption->id]) || !in_array($option, $allowedOptions[$sellingOption->id])) {
							unset($attributeOptions[$key]);
							unset($attributeOptionsIDS[$key]);
						}
					}
					$sellingOption->options = implode("|#", $attributeOptions);
					$sellingOption->optionsIDS = implode("|#", $attributeOptionsIDS);
				}

				foreach ($data['selectedValues'] as $key => $val) {
					$explodeIdVal = explode('_',$key);
					if ($explodeIdVal[1] == $sellingOption->id) {
						$sellingOption->attributeValue = $val;
						break;
					}else{
						$sellingOption->attributeValue = "";
					}
				}
			}else{
				$sellingOption->attributeValue = "";
			}

			$sellingOption->is_mandatory = "1";
			$sellingOption->addColDivs = false;

			$sellingOption->only_for_admin = false;
			$sellingOption->onChangeAction = 'jbdOffers.updateQuantity(this.value,' . $data['offerId'] . ','.$data['mainCatId'].')';

			$itemOtions1 = explode('|#', $sellingOption->optionsIDS);
			foreach ($itemOtions1 as &$opt1) {
				$opt1 = $sellingOption->id . '_' . $opt1;
			}
			$sellingOption->optionsIDS = implode('|#', $itemOtions1);
		}

		//after all options are defined and set for the custom attributes we create the content for them and assign also the
		//quantity select with options defined for it.
		$renderedContent = AttributeService::createStockOptions($sellingOptions, false, array());
		
		$price = "";
		$useStockPrice = "";
		if (count($newSelectedValues) == count($data['selectedValues'])){
			$quantity = $configurations[0];
			
			$price = $quantity->price;
			$useStockPrice = $quantity->use_stock_price;
			
			$selectHtml = '<label for="quantity">'.JText::_("LNG_QUANTITY").'</label>';
			$selectHtml .= '<select onchange="jbdOffers.checkAddToCartStatus()" name="quantity" id="quantity" class="validate[required] select">';
			$selected = false;
			if ($quantity->qty > 0 || $quantity->qty==0) {
				$selectHtml .= '<option selected value="0">0</option>';
				$selected = true;
			}
			$maximum = $quantity->max_sale;
			if ($quantity->qty < $quantity->max_sale) {
				$maximum = $quantity->qty;
			}
			$maximum = ($maximum < MAXIMUM_OFFER_QUANTITY_SELLING) ? $maximum : MAXIMUM_OFFER_QUANTITY_SELLING;
			if ($quantity->qty > 0) {
				for ($i = $quantity->min_sale; $i <= $maximum; $i++) {
					if ($i == $quantity->min_sale && $selected == false) {
						$selectHtml .= '<option selected value="' . $i . '">' . $i . '</option>';
					} else {
						$selectHtml .= '<option value="' . $i . '">' . $i . '</option>';
					}
				}
			}
			$selectHtml .= '</select>';
		}else{
			$selectHtml = '<label for="quantity">'.JText::_("LNG_QUANTITY").'</label>';
			$selectHtml .= '<select onchange="jbdOffers.checkAddToCartStatus()" name="quantity" id="quantity" class="validate[required] select">';
			$selectHtml .= '<option selected value="0">0</option>';
			$selectHtml .= '</select>';
		}
		
		$renderedContent .= '<div class="jbtn-order-item">';
		$renderedContent .= $selectHtml;
		$renderedContent .= '</div>';
		$renderedContent .= '<p class="jbtn-order-info-text"></p>';

		if($useStockPrice && !empty($price)){
			$offerTable = $this->getTable();
			$currency = $offerTable->getOfferCurrency($data['offerId']);
			
			$renderedContent .= '<div class="jbtn-order-item">';
			$renderedContent .= '<div id="offer-stock-price" class="offer-stock-price" unitprice="'.$price.'">'.JText::_("LNG_PRICE").': <span id="total-price">'.$price.'</span> '.$currency->currency_name.'</div>';
			$renderedContent .= '</div>';
		}
		
		return $renderedContent;
	}

	public function checkUserReviews($userId, $itemId) {
		$table = $this->getTable("Review");
		$reviews = $table->getUserReviews($userId, true, $itemId, REVIEW_TYPE_OFFER);

		if (count($reviews)==0) {
			return false;
		}
		return true;
	}

	public function generateCouponCode() {
		$this->offerId = JFactory::getApplication()->input->get('id');

		$orderId = JFactory::getApplication()->input->getInt('orderId',0);

		$user = JBusinessUtil::getUser();
		$offerCouponsTable = JTable::getInstance("OfferCoupon", "JTable");
		$userCoupon = $offerCouponsTable->getUserCoupon($this->offerId, $orderId , $user->id);

		if (!empty($userCoupon)) {
			//if user have a coupon then return it
			return $userCoupon->id;
		} else {
			//if user don`t have a coupon then generate one
			$totalNumber = $offerCouponsTable->getTotalOfferCoupons($this->offerId)+1;

			if ($totalNumber>1) {
				$lastOfferCoupon = $offerCouponsTable->getLastOfferCoupon($this->offerId);
				$prefix = explode('-', $lastOfferCoupon->code)[0];
			} else {
				$prefix = strtoupper(substr(str_shuffle(str_repeat("abcdefghijklmnopqrstuvwxyz", 5)), 0, 3));
			}

			$number = str_pad($totalNumber, 4, '0', STR_PAD_LEFT);
			$code = $prefix."-".$number;

			//If the total number of available coupons in not reached save the coupon
			$offersTable = JTable::getInstance("Offer", "JTable");
			$offer = $offersTable->getOffer($this->offerId);

			$couponId = false;
			if ((($totalNumber-1) < $offer->total_coupons)) {
				$couponId = $offerCouponsTable->saveCoupon($user->id, $this->offerId, $orderId , $code);
			}

			return $couponId;
		}
	}

	public function getCoupon() {
		$couponId = $this->generateCouponCode();

		if ($couponId) {
			$model = $this->getInstance('OfferCoupon', 'JBusinessDirectoryModel');
			$model->show($couponId);
		}
	}

	public function checkOffer() {
		$offerCouponsTable = JTable::getInstance("OfferCoupon", "JTable");
		$checkOffer = $offerCouponsTable->checkOffer($this->offerId);
		
		return $checkOffer;
	}

	public function getOfferAttributes() {
		$attributesTable = $this->getTable('OfferAttributes');
		$categoryId = null;
		if ($this->appSettings->enable_attribute_category) {
			$categoryId = -1;
			if (!empty($this->offer->main_subcategory)) {
				$categoryId = $this->offer->main_subcategory;
			}
		}
		$result = $attributesTable->getOfferAttributes($this->offerId, $categoryId);
		
		return $result;
	}

	/**
	 * Get the offers that are about to expire and send an email to the offer owners
	 */
	public function checkOffersAboutToExpire() {
		$offerTable = $this->getTable("Offer");
		$appSettings = JBusinessUtil::getApplicationSettings();
		$nrDays = $appSettings->expiration_day_notice;
		$offers = $offerTable->getOffersAboutToExpire($nrDays);
		foreach ($offers as $offer) {
			echo "sending expiration e-mail to: ".$offer->subject."<br/>";
			$result = EmailService::sendOfferExpirationEmail($offer, $nrDays);
			if ($result) {
				$offerTable->updateExpirationEmailDate($offer->id);
			}
		}
		exit;
	}

	/**
	 * Get All Offer Reviews
	 * @return mixed
	 */
	public function getReviews($reviewId = null) {
		$reviewsTable = $this->getTable("Review");
		$reviews = $reviewsTable->getReviews($this->offerId, $this->appSettings->show_pending_review, REVIEW_TYPE_OFFER, $reviewId);

		if (!empty($reviews)) {
			foreach ($reviews as $review) {
				$review->responses =  $reviewsTable->getCompanyReviewResponse($review->id);
				if (isset($review->scores)) {
					$review->scores = explode(",", $review->scores);
				}
				if (isset($review->answer_ids)) {
					$review->answerIds = explode(",", $review->answer_ids);
				}
				$review->pictures = $reviewsTable->getReviewPictures($review->id);
			}
		}

		return $reviews;
	}

	/**
	 * Get the offer based on ID
	 * @param $offerid
	 * @return mixed
	 */
	public function getPlainOffer($offerid) {
		$offersTable = $this->getTable("Offer");
		$offer = $offersTable->getOffer($offerid);
		return $offer;
	}

	/**
	 * Save a new review on offer
	 *
	 * @param $data
	 * @return bool|int|JException|null|void
	 */
	public function saveReview($data) {
		$id	= (!empty($data['id'])) ? $data['id'] : (int) $this->getState('review.id');
		$isNew = true;

		$itemId = $data['itemId'];
		$rating = 0;
		if (isset($data["review"])) {
			$rating = $data["review"];
			$data["rating"] = $rating;
		}

		$table = $this->getTable("Review");

		// Load the row if saving an existing item.
		if ($id > 0) {
			$table->load($id);
			$isNew = false;
		}

		// Bind the data.
		if (!$table->bind($data)) {
			$this->setError($table->getError());
		}

		// Check the data.
		if (!$table->check()) {
			$this->setError($table->getError());
		}

		// Store the data.
		if (!$table->store()) {
			$this->setError($table->getError());
		}

		$reviewId = $table->id;

		if (isset($data['pictures']) && count($data['pictures'])>0) {
			$oldId = $isNew?0:$id;
			$this->storePictures($data, $reviewId, $oldId);
		}
		
		if ($this->appSettings->show_pending_review) {
			$table->updateReviewScore($itemId, REVIEW_TYPE_OFFER);
		}
		
		NotificationService::sendReviewNotification($itemId, ITEM_TYPE_OFFER);

		return true;
	}

	/**
	 * Increase the review Like Count
	 * @param $reviewId
	 * @return mixed
	 */
	public function increaseReviewLikeCount($reviewId) {
		$table = $this->getTable("Review");
		return $table->increaseReviewLike($reviewId);
	}

	/**
	 * Increase the review dislike count
	 * @param $reviewId
	 * @return mixed
	 */
	public function increaseReviewDislikeCount($reviewId) {
		$table = $this->getTable("Review");
		return $table->increaseReviewDislike($reviewId);
	}


	/**
	 * Save the review responses
	 *
	 * @param $data
	 * @return bool|int|JException|null|void
	 */
	public function saveReviewResponse($data) {
		//save in banners table
		$row = $this->getTable("reviewresponses");
		$data["state"]=1;

		// Bind the form fields to the table
		if (!$row->bind($data)) {
			dump($row->getError());
			$this->setError($row->getError());
			return false;
		}
		// Make sure the record is valid
		if (!$row->check()) {
			dump($row->getError());
			$this->setError($row->getError());
			return false;
		}

		// Store the web link table to the database
		if (!$row->store()) {
			dump($row->getError());
			$this->setError($row->getError());
			return false;
		}

		$offersTable = JTable::getInstance("Offer", "JTable");
		$offer = $offersTable->getOffer($data['companyId']);
		$companiesTable = JTable::getInstance("Company", "JTable");
		$company = $companiesTable->getCompany($offer->companyId);
		//dump($company);exit;
		$ret = EmailService::sendOfferReviewResponseEmail($offer, $company, $data);

		return $ret;
	}

	/**
	 * Add a report for any review added on an offer
	 * @param $data
	 * @return bool|int|JException|null
	 */
	public function reportAbuse($data) {
		$data["state"]=1;
		$row = $this->getTable("reviewabuses");

		// Bind the form fields to the table
		if (!$row->bind($data)) {
			$this->setError($row->getError());
			return false;
		}
		// Make sure the record is valid
		if (!$row->check()) {
			$this->setError($row->getError());
			return false;
		}

		// Store the web link table to the database
		if (!$row->store()) {
			$this->setError($row->getError());
			return false;
		}

		$offersTable = JTable::getInstance("Offer", "JTable");
		$offer = $offersTable->getOffer($data['companyId']);

		$reviewsTable = $this->getTable("Review");
		$review = $reviewsTable->getReview($data["reviewId"]);

		$companiesTable = JTable::getInstance("Company", "JTable");
		$company = $companiesTable->getCompany($offer->companyId);

		$ret = EmailService::sendOfferReportAbuseEmail($data, $review, $company, $offer);

		return $ret;
	}

	public function contactOfferCompany($data, $offer) {
		$company = $this->getTable("Company");
		$company->load($data['companyId']);

		if (!empty($data['contact_id_offer'])) {
			$company->email = $data['contact_id_offer'];
		}
		
		$data["description"] = nl2br(htmlspecialchars($data["description"], ENT_QUOTES));
		$ret = EmailService::sendOfferContactCompanyEmail($company, $offer, $data);

		return $ret;
	}

	/**
	 * Get all event videos
	 *
	 * @return mixed
	 */
	public function getOfferVideos() {
		$table = $this->getTable("OfferVideos");
		$videos = $table->getOfferVideos($this->offerId);

		if (!empty($videos)) {
			$data = array();
			foreach ($videos as $video) {
				$data = JBusinessUtil::getVideoDetails($video->url);
				$video->url = $data['url'];
				$video->videoType = $data['type'];
				$video->videoId = $data['video-id'];
				$video->videoThumbnail = $data['thumbnail'];
			}
		}

		return $videos;
	}

	/**
	 * Method to increase the view count of the offer, both on the
	 * offer and statistics table
	 *
	 * @param $offerId int ID of the offer
	 * @return bool
	 */
	public function increaseViewCount($offerId) {
		$offersTable = $this->getTable();
		$offersTable->increaseViewCount($offerId);

		// prepare the array with the table fields
		$data = array();
		$data["id"] = 0;
		$data["item_id"] = $offerId;
		$data["item_type"] = STATISTIC_ITEM_OFFER;
		$data["date"] = JBusinessUtil::convertToMysqlFormat(date('Y-m-d')); //current date
		$data["type"] = STATISTIC_TYPE_VIEW;
		$statisticsTable = $this->getTable("Statistics", "JTable");
		if (!$statisticsTable->save($data)) {
			return false;
		}

		return true;
	}


	public function storePictures($data, $reviewId, $oldId) {
		$usedFiles = array();
		if (!empty($data['pictures'])) {
			foreach ($data['pictures'] as $value) {
				array_push($usedFiles, $value["picture_path"]);
			}
		}

		$pictures_path = JBusinessUtil::makePathFile(BD_PICTURES_UPLOAD_PATH);
		$review_pictures_path = JBusinessUtil::makePathFile(BD_REVIEW_PICTURES_PATH.($reviewId)."/");
		JBusinessUtil::removeUnusedFiles($usedFiles, $pictures_path, $review_pictures_path);

		$picture_ids 	= array();
		foreach ($data['pictures'] as $value) {
			$row = $this->getTable('ReviewPictures');

			$pic = new stdClass();
			$pic->id = 0;
			$pic->reviewId = $reviewId;
			$pic->picture_info = $value['picture_info'];
			$pic->picture_path = $value['picture_path'];
			$pic->picture_enable = $value['picture_enable'];

			$pic->picture_path = JBusinessUtil::moveFile($pic->picture_path, $reviewId, $oldId, BD_REVIEW_PICTURES_PATH);

			//dump("save");
			//dbg($pic);
			//exit;
			if (!$row->bind($pic)) {
				throw( new Exception($row->getError()) );
				$this->setError($row->getError());
			}
			// Make sure the record is valid
			if (!$row->check()) {
				throw( new Exception($row->getError()) );
				$this->setError($row->getError());
			}

			// Store the web link table to the database
			if (!$row->store()) {
				throw( new Exception($row->getError()) );
				$this->setError($row->getError());
			}

			$picture_ids[] = $this->_db->insertid();
		}


		$query = " DELETE FROM #__jbusinessdirectory_review_pictures
				WHERE reviewId = '".$reviewId."'
				".(count($picture_ids)> 0 ? " AND id NOT IN (".implode(',', $picture_ids).")" : "");

		//dbg($query);
		//exit;
		$this->_db->setQuery($query);
		try {
			$this->_db->execute();
		} catch (RuntimeException $e) {
			$this->setError($e->getMessage());
			return false;
		}
		//~prepare photos
		//exit;
	}

	public function addBookmark($data) {
		//save in banners table
		$row = $this->getTable("Bookmark");

		// Bind the form fields to the table
		if (!$row->bind($data)) {
			dump($row->getError());
			$this->setError($row->getError());
			return false;
		}
		// Make sure the record is valid
		if (!$row->check()) {
			dump($row->getError());
			$this->setError($row->getError());
			return false;
		}

		// Store the web link table to the database
		if (!$row->store()) {
			dump($row->getError());
			$this->setError($row->getError());
			return false;
		}

		return true;
	}

	public function updateBookmark($data) {
		//save in banners table
		$row = $this->getTable("Bookmark");

		// Bind the form fields to the table
		if (!$row->bind($data)) {
			dump($row->getError());
			$this->setError($row->getError());
			return false;
		}
		// Make sure the record is valid
		if (!$row->check()) {
			dump($row->getError());
			$this->setError($row->getError());
			return false;
		}

		// Store the web link table to the database
		if (!$row->store()) {
			dump($row->getError());
			$this->setError($row->getError());
			return false;
		}

		return true;
	}

	public function removeBookmark($data) {
		$row = $this->getTable("Bookmark");
		return $row->delete($data["id"]);
	}

	public function saveOfferMessages() {
		$data = array();
		$data["name"] = JFactory::getApplication()->input->get('firstName');
		$data["surname"] = JFactory::getApplication()->input->get('lastName');
		$data["email"] = JFactory::getApplication()->input->getString('email');
		$data["message"] = JFactory::getApplication()->input->get('description', '', 'RAW');
		$data["item_id"] = JFactory::getApplication()->input->get('offer_Id');
		$data["type"] = MESSAGE_TYPE_OFFER;
		$data["user_id"] = JBusinessUtil::getUser()->id;
		$data["read"] = '0';
		$table = $this->getTable("Messages");

		// Bind the data.
		if (!$table->bind($data)) {
			$this->setError($table->getError());
			dump($table->getError());
			return false;
		}

		// Check the data.
		if (!$table->check()) {
			$this->setError($table->getError());
			dump($table->getError());
			return false;
		}

		// Store the data.
		if (!$table->store()) {
			$this->setError($table->getError());
			dump($table->getError());
			return false;
		}

		return true;
	}

	/*
	 * Retrieve the currect active package for a listing
	 */
	public function getPackage($companyId=null) {
		if (empty($companyId)) {
			$companyId = $this->companyId;
		}
		$table = $this->getTable("Package");
		$package = $table->getCurrentActivePackage($companyId);

		return $package;
	}
}
?>

