<?php

/**
 * @package    JBusinessDirectory
 *
 * @author CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2019 CMS Junkie. All rights reserved.
 * @license   https://www.gnu.org/licenses/agpl-3.0.en.html
 */

defined('_JEXEC') or die('Restricted access');

use Joomla\Utilities\ArrayHelper;

JTable::addIncludePath('administrator/components/com_jbusinessdirectory/tables');

/**
 * Class OrderService
 */
class OrderService {
	/**
	 * Create the order summary
	 *
	 * @param $orderDetails object containing the order details
	 *
	 * @return string
	 * @since 5.0.1
	 */
	public static function getOrderSummary($orderDetails) {
		$appSettings = JBusinessUtil::getApplicationSettings();
		ob_start();
	?>

		<div class="order-items">
			<div class="">
				<?php echo JText::_('LNG_ORDER_DETAILS'); ?>
			</div>

			<div class="order-price">
				<?php if($orderDetails->package->expiration_type == 4){ ?>
					<?php echo JBusinessUtil::getTrialText($orderDetails->package); ?>
					<div class="order-next-price"><?php echo JText::_("LNG_THEN") ?> <?php echo JBusinessUtil::getPriceFormat($orderDetails->package->price); ?>
						<div class="price-duration"> / <?php echo JBusinessUtil::getPackageDuration($orderDetails->package) ?></div>
					</div>
				<?php }else{ ?>
					<?php echo JBusinessUtil::getPriceFormat($orderDetails->initial_amount); ?>
					<div class="price-duration"> / <?php echo JBusinessUtil::getPackageDuration($orderDetails->package) ?></div>
				<?php } ?>
			</div>

			<div class="order-service">
				<?php echo $orderDetails->service; ?>
			</div>

			<div class="order-details">
				<?php echo $orderDetails->description; ?>
			</div>

			<div class="order-spacer">

			</div>

			<div class="order-section">
				<div class="order-item-title"><?php echo $orderDetails->service ?></div>
				<div class="order-item-desc"><?php echo JBusinessUtil::getPriceFormat($orderDetails->initial_amount) ?></div>
			</div>

			<?php if ($orderDetails->discount_amount > 0) { ?>
				<div class="order-section">
					<div class="order-item-title"><?php echo JText::_("LNG_DISCOUNT") ?></div>
					<div class="order-item-desc"><?php echo JBusinessUtil::getPriceFormat($orderDetails->discount_amount) ?></div>
				</div>
			<?php } ?>

			<div class="order-section">
				<div class="order-item-title"><?php echo JText::_("LNG_SUB_TOTAL") ?></div>
				<div class="order-item-desc"><?php echo JBusinessUtil::getPriceFormat($orderDetails->initial_amount - $orderDetails->discount_amount) ?></div>
			</div>


			<?php if ($orderDetails->vat > 0) { ?>
				<div class="order-section">
					<div class="order-item-title"><?php echo JText::_("LNG_VAT") ?> <span class="text-small">(<?php echo (!empty($orderDetails->vat) ? $orderDetails->vat : $appSettings->vat) . "%" ?>)</span></div>
					<div class="order-item-desc"><?php echo JBusinessUtil::getPriceFormat($orderDetails->vat_amount) ?></div>
				</div>
			<?php } ?>

			<?php
			if (!empty($orderDetails->taxes)) {
				foreach ($orderDetails->taxes as $tax) {
			?>
					<div class="order-section">
						<div class="order-item-title"><?php echo $tax->tax_name ?> <span class="text-small"><?php echo ($tax->tax_type == 2) ? "( " . $tax->tax_amount . " %)" : "" ?></span></div>
						<div class="order-item-desc"><?php echo JBusinessUtil::getPriceFormat($tax->tax_calc_amount) ?></div>
					</div>
				<?php } ?>
			<?php } ?>

			<div class="order-section">
				<div class="order-item-title"><?php echo JText::_("LNG_TOTAL") ?></div>
				<div class="order-item-desc"><?php echo JBusinessUtil::getPriceFormat($orderDetails->amount) ?></div>
			</div>

		</div>

	<?php
		$result = ob_get_contents();
		ob_end_clean();
		return $result;
	}

	public static function getOrderBillingDetailsSummary($order) {
		ob_start();

	?>
		<div class="billing-details">
			<div class="title"><?php echo JText::_("LNG_BILLING_DETAILS") ?></div>
			<div class="detail-spacer"></div>
			<div class="billing-item">
				<div class="billing-item-title"><?php echo JText::_("LNG_FULL_NAME") ?></div>
				<div class="billing-item-desc"><?php echo $order->billingDetails->first_name . ' ' . $order->billingDetails->last_name ?></div>
			</div>
			<?php if(!empty($order->billingDetails->company_name)){ ?>
				<div class="billing-item">
					<div class="billing-item-title"><?php echo JText::_("LNG_BUSINESS_NAME") ?></div>
					<div class="billing-item-desc"><?php echo $order->billingDetails->company_name ?></div>
				</div>
			<?php } ?>

			<?php if(!empty($order->billingDetails->vat_details)){ ?>
				<div class="billing-item">
					<div class="billing-item-title"><?php echo JText::_("LNG_VAT_DETAILS") ?></div>
					<div class="billing-item-desc"><?php echo $order->billingDetails->vat_details ?></div>
				</div>
			<?php } ?>
			<div class="billing-item">
				<div class="billing-item-title"><?php echo JText::_("LNG_ADDRESS") ?></div>
				<div class="billing-item-desc">
					<?php echo $order->billingDetails->address ?>
					<?php echo $order->billingDetails->city . ' ' . $order->billingDetails->postal_code . ', ' . JBusinessUtil::getCountryName($order->billingDetails->country) ?>
				</div>
			</div>
			<div class="billing-item">
				<div class="billing-item-title"><?php echo JText::_("LNG_EMAIL") ?></div>
				<div class="billing-item-desc">
					<?php echo $order->billingDetails->email ?>
				</div>
			</div>
			<div class="billing-item">
				<div class="billing-item-title"><?php echo JText::_("LNG_PHONE") ?></div>
				<div class="billing-item-desc"><?php echo $order->billingDetails->phone ?></div>
			</div>
		</div>
		<?php
		
		$result = ob_get_contents();
		ob_end_clean();
		return $result;
	}

	/**
	 * Retrieve an order from it's transaction ID
	 * 
	 * @param $transactionID string ID of the transaction of the order
	 * @return object
	 * 
	 * @since 5.4.0
	 */
	public static function getOrderByTransactionID($transactionID) {
		$ordersTable = JTable::getInstance("Order", "JTable");
		$order = $ordersTable->getOrderByTransactionID($transactionID);

		return $order;
	}

	/**
	 * Set transaction ID of an order
	 * 
	 * @param $orderId int ID of the order
	 * @param $transactionID string ID of the transaction for the order
	 * 
	 * @since 5.4.0
	 */
	public static function setOrderTransactionId($orderId, $transactionID) {
		$ordersTable = JTable::getInstance("Order", "JTable");
		$ordersTable->setOrderTransactionID($orderId, $transactionID);
	}

	/**
	 * Get the orer
	 *
	 * @param $orderId int ID of the order
	 * @param $countryId int ID of the country
	 *
	 * @return bool
	 *
	 * @since 5.5.0
	 */
	public static function getOrder($orderId) {

		$orderTable = JTable::getInstance("Order", "JTable", array());
		$orderTable->load($orderId);

		$properties = $orderTable->getProperties(1);
		$order = ArrayHelper::toObject($properties, 'JObject');

		return $order;
	}

	/**
	 * Calculates the correct VAT based on country and updates the order on the database.
	 *
	 * @param $orderId int ID of the order
	 * @param $countryId int ID of the country
	 *
	 * @return bool
	 *
	 * @since 5.5.0
	 */
	public static function updateOrderVAT($orderId, $countryId) {
		$orderTable = JTable::getInstance("Order", "JTable", array());
		$orderTable->load($orderId);

		$properties = $orderTable->getProperties(1);
		$order = ArrayHelper::toObject($properties, 'JObject');

		// apply discount (if exists) before applying VAT
		$amount = $order->initial_amount;
		if (!empty($order->discount_amount)) {
			$amount -= $order->discount_amount;
		}

		$vatAmount = TaxService::getVatAmount($amount, $countryId);
		$vat = 0;
		if(!empty($amount)){
			$vat = floatval($vatAmount / $amount) * 100;
		}
		
		if (!$orderTable->updateOrderVAT($orderId, $vatAmount, $vat)) {
			$application = JFactory::getApplication();
			$application->enqueueMessage($orderTable->getDbo()->getError(), 'error');
			return false;
		}

		return true;
	}

	public static function createUserPackageOrder($packageId) {
		$appSettings = JBusinessUtil::getApplicationSettings();
		$user = JBusinessUtil::getUser();
		
		$packageTable = JTable::getInstance("Package", "JTable");
		$package = $packageTable->getPackage($packageId);

		if (empty($package) || $package->price == 0) {
			return false;
		}

		if ($appSettings->enable_multilingual) {
			JBusinessDirectoryTranslations::updateEntityTranslation($package, PACKAGE_TRANSLATION);
		}

		$serviceName = $package->name;

		$orderId = JText::_("LNG_NEW_USER_PACKAGE")." - ".$package->name;
		$description = JText::_("LNG_FROM_USER_PACKAGE_ORDER")." $user->name"." - ".$package->name;

		$data = array();
		$data["start_trial_date"] = null;
		$data["order_id"] = $orderId;
		$data["company_id"] = -1;
		$data["package_id"] = $packageId;
		$appSettings->vat = floatval($appSettings->vat);
		$data["vat"] = $appSettings->vat;
		
		$start_date = date("Y-m-d");
		$remainingAmount = 0;
		
		if (!empty($appSettings->package_date) && $package->expiration_type > 1 && false) {
			// if the package date is set we calculate the remaining days and amount up to the date
			$data["start_trial_date"] =  date("Y-m-d");
			
			//calculate the trial end date
			$timeUnit = JBusinessUtil::getTimeUnit($package->trial_period_unit);
			$endDate = date('Y-m-d', strtotime($start_date. " + $package->trial_period_amount $timeUnit"));
			$data["end_trial_date"] =  $endDate ;

			$remainingDays = floor((strtotime($appSettings->package_date) - strtotime(date("Y-m-d")))/ (60 * 60 * 24));
			
			//calculating the price of period between the package_date and the package actual end date
			$remainingAmount = $package->price - $package->price/$package->days * $remainingDays;
			$remainingAmount = $remainingAmount  + $appSettings->vat*$remainingAmount /100;
			
			if ($package->expiration_type >=3) {
				$start_date = $appSettings->package_date;
				$data["trial_amount"] = $remainingAmount;
				$data["trial_days"] = $remainingDays;
			} else {
				$data["trial_amount"] = 0;
				$data["trial_days"] = 0;
			}
		} 

		$table = JTable::getInstance("Order" , "JTable");

		if (!empty($appSettings->package_date && $package->expiration_type < 3)) {
			$data["initial_amount"] = $package->price - $remainingAmount;
			$data["vat_amount"]     = $appSettings->vat * $data["initial_amount"] / 100;
			$data["amount"]         = $data["initial_amount"] + $data["vat_amount"];
		} else {
			$data["initial_amount"] = $package->price;
			$data["vat_amount"] = $appSettings->vat * $package->price / 100;
			$data["amount"]     = $package->price + $appSettings->vat * $package->price / 100;

			// if ($type == UPDATE_TYPE_EXTEND && !empty($package->renewal_price)) {
			// 	$data["initial_amount"] = $package->renewal_price;
			// 	$data["amount"]     = $package->renewal_price + $appSettings->vat * $package->renewal_price / 100;
			// }
		}

		JModelLegacy::addIncludePath(JPATH_COMPONENT_SITE . '/models', 'BillingDetails');
		$model = JModelLegacy::getInstance('BillingDetails', 'JBusinessDirectoryModel', array('ignore_request' => true));

		$billingDetails = $model->getItem();
		
		$countryId = (!empty($billingDetails) && !empty($billingDetails->id) && !empty($billingDetails->country)) ? $billingDetails->country->id : null;

		$taxObject = TaxService::calculateTaxes($data["amount"], JBD_PACKAGES, $countryId);
		$data["amount"] += (float) $taxObject->tax_amount;

		$timeUnit = "days";
		switch ($package->time_unit) {
			case "Y":
				$timeUnit = "years";
				break;
			case "M":
				$timeUnit = "months";
				break;
			case "D":
				$timeUnit = "days";
				break;
		};

		$timeUnit = JBusinessUtil::getTimeUnit($package->time_unit);
		$endDate = date('Y-m-d', strtotime($start_date. " + $package->time_amount $timeUnit"));
		
		if ($package->expiration_type == 1) {
			$endDate = date('Y-m-d', strtotime($start_date. " + 100 years"));
		}

		$data["state"]            = 0;
		$data["start_date"]       = $start_date;
		$data["end_date"]         = $endDate;
		//disable start and end date for recurring with trial
		
		if ($package->expiration_type == 4) {
			$data["start_date"] = "";
			$data["end_date"] = "";
			
			$data["start_trial_date"] =  $start_date;
			//calculate the trial end date
			$timeUnit = JBusinessUtil::getTimeUnit($package->trial_period_unit);
			$endDate = date('Y-m-d', strtotime($start_date. " + $package->trial_period_amount $timeUnit"));
			$data["end_trial_date"] =  $endDate ;
		} else {
			$data["start_trial_date"] = null;
			$data["end_trial_date"] = null;
		}

		$data["user_name"]        = $user->name;
		$data["user_id"]          = $user->id;
		$data["service"]          = $serviceName;
		$data["description"]      = $description;
		// $data["type"]             = $type;
		$data["currency"]         = $appSettings->currency_name;

		$data["discount_code"]    = null;
		$data["discount_amount"]  = 0;

		// Bind the data.
		if (!$table->bind($data)) {
			JFactory::getApplication()->enqueueMessage($table->getError(), 'error');
			return false;
		}
		
		// Check the data.
		if (!$table->check()) {
			JFactory::getApplication()->enqueueMessage($table->getError(), 'error');
			return false;
		}
		
		// Store the data.
		if (!$table->store()) {
			JFactory::getApplication()->enqueueMessage($table->getError(), 'error');
			return false;
		}
		
		$table->createOrderDetails($table->id, $package->id);
		$table->createOrderTax($table->id, JBD_PACKAGES, $taxObject->taxes);

		return $table->id;
	}
}
