<?php
/**
 * @package    JBusinessDirectory
 *
 * @author CMSJunkie http://www.cmsjunkie.com
 * @copyright  Copyright (C) 2007 - 2019 CMS Junkie. All rights reserved.
 * @license   https://www.gnu.org/licenses/agpl-3.0.en.html
 */
defined('_JEXEC') or die('Restricted access');

use Joomla\Utilities\ArrayHelper;

jimport('joomla.application.component.modeladmin');

/**
 * Announcement Model
 *
 */
class JBusinessDirectoryModelAnnouncement extends JModelAdmin {
	/**
	 * @var        string    The prefix to use with controller messages.
	 * @since   1.6
	 */
	protected $text_prefix = 'COM_JBUSINESSDIRECTORY_ANNOUNCEMENT';

	/**
	 * Model context string.
	 *
	 * @var        string
	 */
	protected $_context = 'com_jbusinessdirectory.announcement';

	public function __construct($config = array()) {
		$this->appSettings = JBusinessUtil::getApplicationSettings();

		parent::__construct($config);
	}

	/**
	 * Method to test whether a record can be deleted.
	 *
	 * @param   object    A record object.
	 *
	 * @return  boolean  True if allowed to delete the record. Defaults to the permission set in the component.
	 */
	protected function canDelete($record) {
		return true;
	}

	/**
	 * Method to test whether a record can be deleted.
	 *
	 * @param   object    A record object.
	 *
	 * @return  boolean  True if allowed to change the state of the record. Defaults to the permission set in the component.
	 */
	protected function canEditState($record) {
		return true;
	}

	/**
	 * Returns a Table object, always creating it
	 *
	 * @param   type    The table type to instantiate
	 * @param   string    A prefix for the table class name. Optional.
	 * @param   array  Configuration array for model. Optional.
	 * @return  JTable    A database object
	 */
	public function getTable($type = 'Announcements', $prefix = 'Table', $config = array()) {
		return JTable::getInstance($type, $prefix, $config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @since   1.6
	 */
	protected function populateState() {
		$app = JFactory::getApplication('administrator');

		// Load the User state.
		$id = JFactory::getApplication()->input->getInt('id');
		$this->setState('announcement.id', $id);
	}

	/**
	 * Method to get a menu item.
	 *
	 * @param   integer    The id of the menu item to get.
	 *
	 * @return  mixed  Menu item data object on success, false on failure.
	 */
	public function &getItem($itemId = null) {
		$itemId = (!empty($itemId)) ? $itemId : (int)$this->getState('announcement.id');
		$false = false;

		// Get a menu item row instance.
		$table = $this->getTable();

		// Attempt to load the row.
		$return = $table->load($itemId);

		// Check for a table object error.
		if ($return === false && $table->getError()) {
			$this->setError($table->getError());
			return $false;
		}

		$properties = $table->getProperties(1);
		$value = ArrayHelper::toObject($properties, 'JObject');

		if ($value->expiration_date == '0000-00-00') {
			$value->expiration_date = null;
		}

		$value->expiration_date = JBusinessUtil::convertToFormat($value->expiration_date);

		$companyTable = $this->getTable('Company', 'JTable');
		if (!empty($value->id) && !empty($value->company_id)) {
			$value->companyOptions = array($companyTable->getCompany($value->company_id));
		}

		return $value;
	}

	/**
	 * Method to get the menu item form.
	 *
	 * @param   array  $data		Data for the form.
	 * @param   boolean	$loadData	True if the form is to load its own data (default case), false if not.
	 * @return  JForm	A JForm object on success, false on failure
	 * @since   1.6
	 */
	public function getForm($data = array(), $loadData = true) {
		exit;
	}


	/**
	 * Method to change the published state of one or more records.
	 *
	 * @param   array    &$pks   A list of the primary keys to change.
	 * @param   integer  $value  The value of the published state.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   2.5
	 */
	public function publish(&$pks, $value = 1) {
		if (parent::publish($pks, $value)) {
			return true;
		}
	}

	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return  mixed  The data for the form.
	 * @since   1.6
	 */
	protected function loadFormData() {
		// Check the session for previously entered form data.
		$data = JFactory::getApplication()->getUserState('com_jbusinessdirectory.edit.announcement.data', array());

		if (empty($data)) {
			$data = $this->getItem();
		}

		return $data;
	}


	/**
	 * Method to save the form data.
	 *
	 * @param   array  The form data.
	 * @return  boolean  True on success.
	 */
	public function save($data) {
		$id	= (!empty($data['id'])) ? $data['id'] : (int) $this->getState('announcement.id');
		$isNew = true;

		$defaultLng = JBusinessUtil::getLanguageTag();
		$title = JFactory::getApplication()->input->get('title_'.$defaultLng, '', 'RAW');
		$description = JFactory::getApplication()->input->get('description_'.$defaultLng, '', 'RAW');
		$buttonText = JFactory::getApplication()->input->get('button_text_'.$defaultLng, '', 'RAW');

		if (!empty($buttonText) && empty($data["button_text"])) {
			$data["button_text"] = $buttonText;
		}

		if (!empty($title) && empty($data["title"])) {
			$data["title"] = $title;
		}

		if (!empty($description) && empty($data["description"])) {
			$data["description"] = $description;
		}

		if (!isset($data["description"]) && empty($description)) {
			$data["description"] = "";
		}

		if (!empty($data['button_link'])) {
			if (!preg_match("~^(?:f|ht)tps?://~i", $data['button_link'])) {
				$data['button_link'] = "http://" . $data['button_link'];
			}
		}

		if (empty($data["expiration_date"])) {
			$data["expiration_date"] = null;
		}

		if (!empty($data["expiration_date"]) && $data["expiration_date"]!= '0000-00-00 00:00:00') {
			$data["expiration_date"] = JBusinessUtil::convertToMysqlFormat($data["expiration_date"]);
		}

		// Get a row instance.
		$table = $this->getTable();

		// Load the row if saving an existing item.
		if ($id > 0) {
			$table->load($id);
			$isNew = false;
		}

		// Bind the data.
		if (!$table->bind($data)) {
			$this->setError($table->getError());
			return false;
		}

		// Check the data.
		if (!$table->check()) {
			$this->setError($table->getError());
			return false;
		}

		// Store the data.
		if (!$table->store()) {
			$this->setError($table->getError());
			return false;
		}

		if ($this->appSettings->enable_multilingual) {
			JBusinessDirectoryTranslations::saveTranslations(ANNOUNCEMENT_DESCRIPTION_TRANSLATION, $table->id, 'description_');
		}

		$this->setState('announcement.id', $table->id);
		$this->setState('managecompanyannouncement.id', $table->id);

		// Clean the cache
		$this->cleanCache();

		return true;
	}


	/**
	 * Method to delete groups.
	 *
	 * @param   array  An array of item ids.
	 * @return  boolean  Returns true on success, false on failure.
	 */
	public function delete(&$itemIds) {
		// Sanitize the ids.
		$itemIds = (array) $itemIds;
		ArrayHelper::toInteger($itemIds);

		// Get a group row instance.
		$table = $this->getTable();

		// Iterate the items to delete each one.
		foreach ($itemIds as $itemId) {
			if (!$table->delete($itemId)) {
				$this->setError($table->getError());
				return false;
			}
		}

		// Clean the cache
		$this->cleanCache();

		return true;
	}

	public function changeState($id, $value) {
		$table = $this->getTable();
		return $table->changeState($id, $value);
	}
}
